import numpy as np
import networkx as nx
from itertools import chain, combinations

# Generate the power set of a given iterable


def powerset(iterable):
    """Return the power set of the given iterable."""
    s = list(iterable)
    return chain.from_iterable(combinations(s, r) for r in range(len(s) + 1))

# Generate a random submodular function


def generate_random_submodular_function(num_elements):
    """
    Construct a submodular function F using the cut function
    on a randomly generated graph with two additional special nodes s and t.
    """
    # Generate a random graph (Erdős–Rényi model) with random edge weights
    G = nx.erdos_renyi_graph(
        num_elements, p=0.5, seed=np.random.randint(0, 1000))
    for (u, v) in G.edges():
        G.edges[u, v]['weight'] = np.random.uniform(0, 1)

    # Add two special nodes s and t
    s, t = num_elements, num_elements + 1
    G.add_node(s)
    G.add_node(t)

    # Add random edges from s and to t with random weights
    for node in range(num_elements):
        if np.random.rand() < 0.5:
            G.add_edge(s, node, weight=np.random.uniform(0, 1))
        if np.random.rand() < 0.5:
            G.add_edge(node, t, weight=np.random.uniform(0, 1))

    # Define the cut function
    def cut_function(S):
        cut_value = 0
        for u, v, data in G.edges(data=True):
            if (u in S and v not in S) or (v in S and u not in S):
                cut_value += data['weight']
        return cut_value

    # Compute cut values for all subsets and normalize
    V = list(range(num_elements))
    subsets = list(powerset(V))
    cut_values = np.array([cut_function(set(S) | {s}) for S in subsets])
    max_cut_value = cut_values.max() if cut_values.size > 0 else 1

    # Define the normalized submodular function F
    def F(S):
        S_set = set(S)
        # F(S) = normalized cut_function(S ∪ {s})
        return cut_function(S_set | {s}) / max_cut_value if max_cut_value > 0 else 0

    return F
