import numpy as np


def linear_submodular_minimization(F, n, T, a):
    """
     Submodular function minimization with linear transfer sunction.

    Args:
        F (callable): Submodular function.
        n (int): Number of elements in the ground set.
        T (int): Number of iterations.
        a (float): Linear transfer parameter.

    Returns:
        set: The solution set S_T.
    """
    # Initialization
    w = np.full(n, 0.5)              # Initial weight vector
    eta = a / (2 * np.sqrt(n * T))   # Step size
    w_history = np.zeros((T, n))     # Store weight vectors

    for t in range(T):
        # Sort indices in descending order of weights
        pi = np.argsort(-w)

        # Randomly select an index i
        i = np.random.randint(n)

        # Construct subsets B_i and B_{i-1}
        B_i = set(pi[:i+1])
        B_i_minus_1 = set(pi[:i])

        # Simulate oracle feedback (dueling comparison)
        delta_F = F(B_i) - F(B_i_minus_1)
        p = np.clip(delta_F, -1, 1)
        o_t = 1 if np.random.random() < (1 + p) / 2 else -1

        # Gradient estimate
        g_t = np.zeros(n)
        g_t[pi[i]] = o_t * n/a

        # Update and project onto [0,1]^n
        w -= eta * g_t
        np.clip(w, 0, 1)  # projection (in-place)

        # Record current weights
        w_history[t] = w

    # Compute averaged solution
    w_bar = np.mean(w_history, axis=0)

    # Random threshold z ~ U[0,1]
    z = np.random.uniform(0, 1)

    # Construct final set S_T
    S_T = {i for i in range(n) if w_bar[i] >= z}

    return S_T
