import anthropic
import threading
import time
from typing import List, Dict
from concurrent.futures import ThreadPoolExecutor

class Claude_Client:
    def __init__(self, url, api_key: str, model: str = "gpt-4o-mini", max_workers: int = 16):
        self.api_key = api_key
        self.url = url
        self.model = model
        
        self.client = anthropic.Anthropic(
            api_key=self.api_key,
            base_url = self.url,
        )

        self.start_time = time.time()  
        self.usage = {
            "completion_tokens": 0,
            "prompt_tokens": 0,
            "total_tokens": 0
        }
        self.executor = ThreadPoolExecutor(max_workers=max_workers) 
        self.lock = threading.Lock()

    def get_elapsed_time(self) -> float:
        return time.time() - self.start_time

    def fetch_response(self, prompt: str) -> str:
        try:
            response = self.client.messages.create(
                model=self.model,
                max_tokens=512,
                temperature=0.5,
                timeout=60,
                messages=[
                    {"role": "user", "content": prompt}
                ]
            )

            self.update_usage(response.usage)
            return response.content[0].text
        except Exception as e:
            print(f"error: {e}")
            return ""
            

    def update_usage(self, usage: Dict[str, int]):
        with self.lock:
            self.usage['completion_tokens'] += usage.output_tokens
            self.usage['prompt_tokens'] += usage.input_tokens
            self.usage['total_tokens'] += usage.input_tokens+usage.output_tokens

    def get_usage(self) -> Dict[str, int]:
        return self.usage

    def ask_question(self, prompt: str) -> str:
        return self.fetch_response(prompt)

    def ask_questions(self, prompts: List[str]) -> List[str]:
        futures = {self.executor.submit(self.ask_question, prompt): prompt for prompt in prompts}
        
        results = []
        for future in futures:
            result = future.result() 
            #print(result)
            results.append(result)  

        return results

if __name__ == "__main__":
    prompt1 = """
    who are you?
    """
    prompt2 = """
    how can you help?
    """
    model = "claude-3-5-sonnet-20240620"
    import argparse
    parser = argparse.ArgumentParser()
    parser.add_argument("--api_key", type = str, required=True)
    parser.add_argument("--url", type = str, required=True)
    args = parser.parse_args()
    api_key = args.api_key
    url = args.url
    client = Claude_Client(url, api_key, model, max_workers=16)  
    responses = client.ask_questions([prompt1, prompt2]) 
    for response in responses:
        print(response.replace("**", "").replace("### ", "").replace("\n\n", "\n"))
        print("\n")
    print("Token usage:", client.get_usage())
    print(f"Object has been alive for {client.get_elapsed_time()} seconds")