import numpy as np
from torchvision.datasets import CIFAR10, CIFAR100
import os

def generate_mixed_noise_cifar10(delta_uniform=0.1, delta_gaussian=0.05, k=5, save_path='cifar10_mixed_noise.npz', random_seed=42):

    np.random.seed(random_seed)

    cifar10_train = CIFAR10(root='solo-learn/datasets/CIFAR10', train=True, download=True)
    original_images = cifar10_train.data  # (50000, 32, 32, 3), uint8 [0, 255]
    original_labels = np.array(cifar10_train.targets)  # (50000,)
    
    noisy_images = []
    noisy_labels = []

    for img_idx, (img, label) in enumerate(zip(original_images, original_labels)):
        img_normalized = img.astype(np.float32) / 255.0 
        
        for noise_idx in range(k):
            np.random.seed(random_seed + img_idx * k + noise_idx)
            uniform_noise = np.random.uniform(
                low=-delta_uniform, 
                high=delta_uniform, 
                size=img_normalized.shape
            )
            
            gaussian_noise = np.random.normal(
                loc=0.0, 
                scale=delta_gaussian, 
                size=img_normalized.shape
            )
            
            mixed_noise = uniform_noise + gaussian_noise
            noisy_img = np.clip(img_normalized + mixed_noise, 0.0, 1.0)
            
            noisy_img = (noisy_img * 255).astype(np.uint8)
            noisy_images.append(noisy_img)
            noisy_labels.append(label)
    
    noisy_images = np.stack(noisy_images)  # (50000*k, 32, 32, 3)
    noisy_labels = np.array(noisy_labels)  # (50000*k,)
    
    os.makedirs(os.path.dirname(save_path), exist_ok=True)
    np.savez(
        save_path,
        image=noisy_images,
        label=noisy_labels
    )

if __name__ == '__main__':
    generate_mixed_noise_cifar10(
        delta_uniform=0.0015,   
        delta_gaussian=0.001,  
        k=30,                 
        save_path='solo-learn/datasets/extra_data/cifar10_mixed_noise.npz',
        random_seed=42        
    )

