import utils.path_utils
import requests
import json
import tqdm
import os


def fetch_polymarket_resolved_binary(max_questions=1000):
    if os.environ.get("MAX_QUESTIONS"):
        max_questions = int(os.environ.get("MAX_QUESTIONS")) // 2

    url = "https://gamma-api.polymarket.com/markets"
    params = {"closed": "true", "limit": max_questions}
    response = requests.get(url, params=params)

    with open("data/tmp/polymarket_response.json", "w") as f:
        json.dump(response.json(), f, indent=4)

    markets = response.json()

    data = []
    for m in markets:
        if len(eval(m.get("outcomes", []))) == 2:
            outcomes = eval(m.get("outcomes", []))
            outcomePrices = eval(m.get("outcomePrices", []))
            outcomePrices = [float(price) for price in outcomePrices]

            try:
                resolution_index = outcomePrices.index(max(outcomePrices))
                resolution = outcomes[resolution_index]
            except ValueError:
                continue

            if outcomePrices[resolution_index] < 0.99:
                continue

            market_data = {
                "id": f"polymarket-{m['id']}",
                "source": "polymarket",
                "title": m.get("question", m.get("title")),
                "description": m.get("description"),
                "category": m.get("category"),
                "endTime": m.get("endTime"),
                "volume": eval(m.get("volume")),
                "liquidity": eval(m.get("liquidity")),
                "outcomes": outcomes,
                "outcomePrices": outcomePrices,
                "resolution": resolution,
                "marketType": m.get("marketType"),
                "marketMakerAddress": m.get("marketMakerAddress"),
                "updatedBy": m.get("updatedBy"),
            }
            data.append(market_data)

    print(f"Fetched {len(data)} resolved binary markets on polymarket")
    return data


def fetch_metaculus_resolved_binary(max_questions=1000):
    if os.environ.get("MAX_QUESTIONS"):
        max_questions = int(os.environ.get("MAX_QUESTIONS")) // 2

    if os.path.exists("data/tmp/metaculus_response.json"):
        with open("data/tmp/metaculus_response.json", "r") as f:
            questions = json.load(f)
    else:
        questions = []

    if len(questions) < max_questions:
        # https://www.metaculus.com/api/posts/?statuses=resolved&forecast_type=binary&order_by=vote_score&scheduled_resolve_time__gt=2024-10-01
        url = "https://www.metaculus.com/api/posts/"
        params = {
            "statuses": "resolved",
            "forecast_type": "binary",
            "order_by": "-vote_score",
            "scheduled_resolve_time__gt": "2024-10-01",
        }
        with tqdm.tqdm(total=max_questions) as pbar:
            pbar.update(len(questions))
            response = requests.get(url, params=params).json()
            next_page = response["next"]
            questions.extend(response["results"])  # Assuming list format
            pbar.update(len(response["results"]))

            while next_page and len(questions) < max_questions:
                response = requests.get(next_page).json()
                next_page = response["next"]
                questions.extend(response["results"])
                pbar.update(len(response["results"]))

    with open("data/tmp/metaculus_response.json", "w") as f:
        json.dump(questions, f, indent=4)

    # Name fields to match polymarket
    data = [
        {
            "id": f"metaculus-{q['id']}",
            "source": "metaculus",
            "title": q["title"],
            "description": q["question"]["description"],
            "category": (
                q["projects"]["category"][0]["name"]
                if "projects" in q
                and "category" in q["projects"]
                and q["projects"]["category"]
                else None
            ),
            "endDate": q["scheduled_resolve_time"],
            "volume": q["forecasts_count"],
            "score": q["vote"]["score"],
            "outcomes": q["question"]["options"] or ["yes", "no"],
            "resolution": q["question"]["resolution"],
            "marketType": q["question"]["type"],
            "marketMakerAddress": q["author_id"],
        }
        for q in questions
        if "question" in q and q["question"]["resolution"] != "annulled"
    ]

    print(f"Fetched {len(data)} resolved binary markets on metaculus")
    return data


if __name__ == "__main__":

    # Fetch and save
    polymarket_data = fetch_polymarket_resolved_binary()
    with open("data/questions/polymarket_resolved_binary.json", "w") as f:
        json.dump(polymarket_data, f, indent=4)

    metaculus_data = fetch_metaculus_resolved_binary()
    with open("data/questions/metaculus_resolved_binary.json", "w") as f:
        json.dump(metaculus_data, f, indent=4)
