import neal
import neal.simulated_annealing as sa
import dimod
import numpy as np

import generate_dWPE

import itertools


def generate_sk(N):
    # Initialize linear biases (fields) to zero (h vector)
    h = np.zeros(N)
    
    # Generate a random symmetric coupling matrix from N(0, 1)
    J = np.sign(np.random.normal(0, 1, size=(N, N)))
    J = np.triu(J) - np.diag(np.diag(J))
    J = J + J.T
    
    return h, J

def ising_hamiltonian(config, h, J):
    config = np.array(config)
    energy = -np.dot(h, config) - 0.5 * np.dot(config, np.dot(J, config))
    return energy

def brute_force(h, J):
    N = len(h)  # Number of spins
    
    # Generate all 2^N possible configurations of spins (each spin is either -1 or +1)
    spin_configs = itertools.product([-1, 1], repeat=N)
    
    # Initialize variables to store the best (lowest-energy) configuration
    ground_state = None
    ground_energy = float('inf')
    
    # Iterate through all possible configurations
    for config in spin_configs:
        energy = ising_hamiltonian(config, h, J)
        
        # If we find a new lower energy, update the ground state
        if energy < ground_energy:
            ground_energy = energy
            ground_state = config
    
    return ground_state, ground_energy

class SA:

    
    def __init__(self, N, J, H0, prec):
            
        self.J = J
        self.N = np.shape(J)[0]
        h = np.zeros(self.N)
        self.H0 = H0
        self.prec = prec
        
        #default parameteres
        self.bqm = dimod.BQM(h, -J, 'SPIN')
        beta_start, beta_end = neal.default_beta_range(self.bqm)
        
        self.beta_start = beta_start
        self.beta_end = beta_end
        
    def get_default_beta(self):
        return self.beta_start, self.beta_end
        
    def run_SA_neal(self,beta_start,beta_end,K,T):
    
        sampler = neal.SimulatedAnnealingSampler()

        # Run the simulated annealing process
        response = sampler.sample(self.bqm, beta_range=(beta_start, beta_end), num_sweeps=T, num_reads=K)

        samples = []
        energies = []
        energies2 = []
        for sample, energy in zip(response.samples(),response.data_vectors['energy']):
            s = np.array(list(sample.values())).astype(float)
            energy2 = - 0.5 * np.sum(s * (self.J @ s))
            energy2 = np.floor(energy2/self.prec)
            energies2.append(energy2)
            #print(s)
            #print(energy/2,energy2,H0)
            
            samples.append(s.tolist())
            #energies.append(energy/2)
            energies.append(np.floor(energy/2/self.prec))
            
        #return energies, samples
        return energies2, samples

if __name__ == "__main__":
    
    T = 1000     # T is the number of sweeps!
    K = 50
    N = 11
    
    i = int(100000* np.random.rand())
    alpha = float("0.8")
    M = int(N*alpha)
    
    # unbiased wishart for tests
    if 1:
        data = {}
        data['D_WPE'] = 1 # 3
        data['R_WPE'] = -1 # 6
        data['bias'] = 0.0
        J, H0, gs = generate_dWPE.gen_dWPE(i, N, M, data['D_WPE'], data['R_WPE'])
        eps0 = np.mean(np.abs(J))
        prec = 10**(-6) #precision for GS energy
        H0 = np.floor(H0/prec)
        h = np.zeros(N)
    else:
        h, J = generate_sk(N)
        sig, H0 = brute_force(h, J)
        prec = 1
    
    solver = SA(N, J, H0, prec)

    beta_start,beta_end = solver. get_default_beta()
    beta_start = beta_start
    beta_end = beta_end
    
    energies, solutions = solver.run_SA_neal(beta_start,beta_end,K,T)

    print(energies-H0)