## ui.py

import pygame
from constants import Constants

class UI:
    def __init__(self):
        pygame.init()
        self._screen = pygame.display.set_mode((Constants.WINDOW_WIDTH, Constants.WINDOW_HEIGHT))
        pygame.display.set_caption(Constants.GAME_TITLE)
        self._font = pygame.font.SysFont(Constants.FONT_NAME, Constants.FONT_SIZE)
        self.undo_button_rect = pygame.Rect((Constants.WINDOW_WIDTH // 2 - 50, Constants.WINDOW_HEIGHT - Constants.FONT_SIZE - 5), (100, Constants.FONT_SIZE + 10))

    def _draw_tile(self, value, x, y):
        color = Constants.TILE_COLORS.get(value, (0, 0, 0))
        tile_rect = pygame.Rect(x, y, Constants.TILE_SIZE, Constants.TILE_SIZE)
        pygame.draw.rect(self._screen, color, tile_rect)
        if value:
            text_color = (255, 255, 255) if value > 4 else (0, 0, 0)
            text_surface = self._font.render(str(value), True, text_color)
            text_rect = text_surface.get_rect(center=tile_rect.center)
            self._screen.blit(text_surface, text_rect)

    def draw_grid(self, grid: list):
        self._screen.fill(Constants.BACKGROUND_COLOR)
        for i, row in enumerate(grid):
            for j, value in enumerate(row):
                x = j * Constants.TILE_SIZE + (j + 1) * Constants.TILE_MARGIN
                y = i * Constants.TILE_SIZE + (i + 1) * Constants.TILE_MARGIN
                self._draw_tile(value, x, y)

    def display_score(self, current_score: int, high_score: int):
        score_surface = self._font.render(f'Score: {current_score}', True, (255, 255, 255))
        high_score_surface = self._font.render(f'High Score: {high_score}', True, (255, 255, 255))
        self._screen.blit(score_surface, (Constants.TILE_MARGIN, Constants.WINDOW_HEIGHT - 2 * Constants.FONT_SIZE))
        self._screen.blit(high_score_surface, (Constants.WINDOW_WIDTH // 2, Constants.WINDOW_HEIGHT - 2 * Constants.FONT_SIZE))

    def show_game_over(self):
        game_over_surface = self._font.render('Game Over!', True, (255, 0, 0))
        game_over_rect = game_over_surface.get_rect(center=(Constants.WINDOW_WIDTH // 2, Constants.WINDOW_HEIGHT // 2))
        self._screen.blit(game_over_surface, game_over_rect)

    def show_undo_button(self):
        undo_surface = self._font.render('Undo', True, (255, 255, 255))
        pygame.draw.rect(self._screen, (0, 0, 0), self.undo_button_rect.inflate(20, 10), 2)
        self._screen.blit(undo_surface, self.undo_button_rect)

    def update_display(self):
        pygame.display.flip()

    def draw(self, grid: list, current_score: int, high_score: int, game_over: bool):
        self.draw_grid(grid)
        self.display_score(current_score, high_score)
        self.show_undo_button()
        if game_over:
            self.show_game_over()
        self.update_display()

    def handle_undo_button_click(self, game):
        mouse_pos = pygame.mouse.get_pos()
        if self.undo_button_rect.collidepoint(mouse_pos):
            if pygame.mouse.get_pressed()[0]:  # Left mouse button
                game.undo()
                self.draw(game._grid, game._current_score, game._high_score, game.check_game_over())
