## document_parser.py
import re
from collections import Counter
from typing import Dict
import logging

# Configure logging
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')

class DocumentParser:
    def __init__(self):
        # Constructor for DocumentParser, no initialization required for this example
        pass

    def parse_document(self, file_path: str) -> Dict[str, int]:
        """
        Parses the document at the given file path and returns a frequency dictionary
        of words found in the document.

        :param file_path: The path to the document file to be parsed.
        :return: A dictionary with words as keys and their frequencies as values.
        """
        try:
            with open(file_path, 'r', encoding='utf-8') as file:
                content = file.read()
                # Remove punctuation and numbers, convert to lowercase, and find whole words
                content = re.sub(r'[^\w\s]', '', content).lower()
                word_list = re.findall(r'\b\w+\b', content)
                word_frequencies = Counter(word_list)
                return dict(word_frequencies)
        except FileNotFoundError:
            logging.error(f"The file at {file_path} was not found.")
            return {}
        except (IOError, UnicodeDecodeError) as e:
            logging.error(f"An error occurred while parsing the document: {e}")
            return {}
        except Exception as e:
            logging.error(f"An unexpected error occurred: {e}")
            return {}
