## ui_manager.py
import tkinter as tk
from tkinter import filedialog, simpledialog, messagebox
from PIL import Image, ImageTk
from typing import Dict, Callable, Any

class UIManager:
    def __init__(self, generate_word_cloud_callback: Callable[[Dict[str, Any]], Image.Image]):
        """
        UIManager constructor.

        :param generate_word_cloud_callback: A callback function to generate and return the word cloud image.
        """
        self.root = tk.Tk()
        self.generate_word_cloud_callback = generate_word_cloud_callback
        self.file_path = None
        self.settings = {}
        self.word_cloud_image = None  # Add this line
        self.root.title("Word Cloud Generator")

        # Set up the UI components
        self.setup_ui()

    def setup_ui(self):
        """
        Sets up the user interface components.
        """
        # Button to select a document file
        self.select_button = tk.Button(self.root, text="Select Document", command=self.get_user_input)
        self.select_button.pack()

        # Button to generate the word cloud
        self.generate_button = tk.Button(self.root, text="Generate Word Cloud", command=self.on_generate_word_cloud)
        self.generate_button.pack()

        # Canvas to display the word cloud image
        self.canvas = tk.Canvas(self.root, width=800, height=600)
        self.canvas.pack()

        # Scrollbars for the canvas
        self.v_scrollbar = tk.Scrollbar(self.root, orient='vertical', command=self.canvas.yview)
        self.v_scrollbar.pack(side='right', fill='y')
        self.h_scrollbar = tk.Scrollbar(self.root, orient='horizontal', command=self.canvas.xview)
        self.h_scrollbar.pack(side='bottom', fill='x')

        self.canvas.configure(yscrollcommand=self.v_scrollbar.set, xscrollcommand=self.h_scrollbar.set)

    def get_user_input(self) -> Dict[str, Any]:
        """
        Gets user input for the file path and word cloud settings.

        :return: A dictionary containing the file path and settings.
        """
        self.select_document()
        if self.file_path:
            self.settings = self.prompt_for_settings()
        return {
            'file_path': self.file_path,
            'settings': self.settings
        }

    def select_document(self):
        """
        Opens a file dialog to select a document and stores the file path.
        """
        file_path = filedialog.askopenfilename()
        if file_path:
            self.file_path = file_path

    def prompt_for_settings(self) -> Dict[str, Any]:
        """
        Prompts the user for settings and returns them.

        :return: A dictionary of settings.
        """
        width = simpledialog.askinteger("Input", "Word Cloud Width:", minvalue=200, maxvalue=1600, initialvalue=800)
        height = simpledialog.askinteger("Input", "Word Cloud Height:", minvalue=200, maxvalue=1200, initialvalue=600)
        max_words = simpledialog.askinteger("Input", "Maximum Words:", minvalue=10, maxvalue=1000, initialvalue=200)
        background_color = simpledialog.askstring("Input", "Background Color:", initialvalue='white')

        return {
            'width': width if width is not None else 800,
            'height': height if height is not None else 600,
            'max_words': max_words if max_words is not None else 200,
            'background_color': background_color if background_color else 'white'
        }

    def on_generate_word_cloud(self):
        """
        Gathers user input, generates the word cloud image, and calls the display callback.
        """
        if self.file_path and self.settings:
            word_cloud_image = self.generate_word_cloud_callback({'file_path': self.file_path, 'settings': self.settings})
            self.display_word_cloud(word_cloud_image)
        else:
            messagebox.showinfo("Error", "Please select a document and set the preferences.")

    def display_word_cloud(self, word_cloud: Image.Image):
        """
        Displays the word cloud image on the canvas.

        :param word_cloud: The PIL Image object of the generated word cloud.
        """
        self.word_cloud_image = ImageTk.PhotoImage(word_cloud)
        self.canvas.config(scrollregion=(0, 0, word_cloud.width, word_cloud.height))
        self.canvas.create_image(0, 0, anchor=tk.NW, image=self.word_cloud_image)
        self.canvas.config(width=word_cloud.width, height=word_cloud.height)

    def run(self):
        """
        Runs the main loop of the tkinter UI.
        """
        self.root.mainloop()
