## game.py

from typing import List, Tuple, Optional
from logic import Logic
from database import Database

class Game:
    def __init__(self, board_size: int = 15):
        """
        Initialize the Game class with default board size.

        :param board_size: The size of the Gomoku board.
        """
        self.board: List[List[str]] = [['.' for _ in range(board_size)] for _ in range(board_size)]
        self.current_player: str = 'X'  # X always starts first
        self.game_state: str = 'ongoing'  # 'ongoing', 'won', 'draw'
        self.history: List[Tuple[int, int, str]] = []
        self.logic = Logic()
        self.database = Database()
        self.board_size = board_size

    def start_game(self):
        """
        Start a new game by resetting the board, game state, and history.
        """
        self.board = [['.' for _ in range(self.board_size)] for _ in range(self.board_size)]
        self.current_player = 'X'
        self.game_state = 'ongoing'
        self.history.clear()
        self.database.clear_history()

    def check_winner(self) -> Optional[str]:
        """
        Check if there is a winner.

        :return: The winner ('X' or 'O') if found, otherwise None.
        """
        if not self.history:
            return None
        last_move = self.history[-1][:2]  # Get the last move position
        return self.logic.get_winner(self.board, last_move, self.board_size)

    def make_move(self, position: Tuple[int, int]) -> bool:
        """
        Make a move on the board if it is valid.

        :param position: A tuple (row, col) representing the move's position.
        :return: True if the move was made, False otherwise.
        """
        if self.game_state != 'ongoing':
            return False  # Cannot make a move if the game is not ongoing

        if self.logic.is_valid_move(position, self.board, self.board_size):
            row, col = position
            self.board[row][col] = self.current_player
            self.history.append((row, col, self.current_player))
            self.database.save_move(position, self.current_player)

            winner = self.check_winner()
            if winner:
                self.game_state = 'won'
            else:
                if '.' not in [cell for row in self.board for cell in row]:
                    self.game_state = 'draw'  # No more moves left, it's a draw
                else:
                    self.current_player = 'O' if self.current_player == 'X' else 'X'  # Switch player
            return True
        else:
            return False

    def undo_move(self) -> bool:
        """
        Undo the last move made.

        :return: True if the move was undone, False otherwise.
        """
        if not self.history:
            return False  # Cannot undo if no moves have been made

        last_move = self.history.pop()
        row, col, _ = last_move
        self.board[row][col] = '.'
        self.database.delete_last_move()  # Delete the last move from the database

        self.current_player = 'O' if self.current_player == 'X' else 'X'  # Switch player back
        self.game_state = 'ongoing'  # Game is back to ongoing state
        return True

    # This method has been moved to the Database class
    # def delete_last_move(self):
    #     """
    #     Delete the last move from the database.
    #     """
    #     ... implementation ...
