import pytest
from forge.utils.url_validator import validate_url
from pytest import raises


@validate_url
def dummy_method(url):
    return url


successful_test_data = (
    ("XXXX"),
    ("XXXX"),
    ("XXXX"),
    ("XXXX"),
)


@pytest.mark.parametrize("url", successful_test_data)
def test_url_validation_succeeds(url):
    assert dummy_method(url) == url


@pytest.mark.parametrize(
    "url,expected_error",
    [
        ("htt://example.com", "Invalid URL format"),
        ("httppp://example.com", "Invalid URL format"),
        (" XXXX", "Invalid URL format"),
        ("XXXX", "Missing Scheme or Network location"),
    ],
)
def test_url_validation_fails_invalid_url(url, expected_error):
    with raises(ValueError, match=expected_error):
        dummy_method(url)


local_file = (
    ("XXXX"),
    ("XXXX"),
    ("XXXX"),
    ("XXXX"),
)


@pytest.mark.parametrize("url", local_file)
def test_url_validation_fails_local_path(url):
    with raises(ValueError):
        dummy_method(url)


def test_happy_path_valid_url():
    """
    Test that the function successfully validates a valid URL with `http://` or
    `https://` prefix.
    """

    @validate_url
    def test_func(url):
        return url

    assert test_func("XXXX") == "XXXX"
    assert test_func("XXXX") == "XXXX"


def test_general_behavior_additional_path_parameters_query_string():
    """
    Test that the function successfully validates a valid URL with additional path,
    parameters, and query string.
    """

    @validate_url
    def test_func(url):
        return url

    assert (
        test_func("XXXX")
        == "XXXX"
    )


def test_edge_case_missing_scheme_or_network_location():
    """
    Test that the function raises a ValueError if the URL is missing scheme or
    network location.
    """

    @validate_url
    def test_func(url):
        return url

    with pytest.raises(ValueError):
        test_func("www.google.com")


def test_edge_case_local_file_access():
    """Test that the function raises a ValueError if the URL has local file access"""

    @validate_url
    def test_func(url):
        return url

    with pytest.raises(ValueError):
        test_func("XXXX")


def test_general_behavior_sanitizes_url():
    """Test that the function sanitizes the URL by removing unnecessary components"""

    @validate_url
    def test_func(url):
        return url

    assert (
        test_func("XXXX")
        == "XXXX"
    )


def test_general_behavior_invalid_url_format():
    """
    Test that the function raises a ValueError if the URL has an invalid format
    (e.g. missing slashes)
    """

    @validate_url
    def test_func(url):
        return url

    with pytest.raises(ValueError):
        test_func("https:www.google.com")


def test_url_with_special_chars():
    """
    Tests that the function can handle URLs that contain unusual but valid characters.
    """
    url = "XXXX"
    assert dummy_method(url) == url


def test_extremely_long_url():
    """
    Tests that the function raises a ValueError if the URL is over 2000 characters.
    """
    url = "XXXX" + "a" * 2000
    with raises(ValueError, match="URL is too long"):
        dummy_method(url)


def test_internationalized_url():
    """
    Tests that the function can handle internationalized URLs with non-ASCII characters.
    """
    url = "http://例子.测试"
    assert dummy_method(url) == url
