import torch
import torch.nn as nn
from causally.model.utils import get_linear_layers
from causally.model.abstract_model import AbstractModel

class TARNet(AbstractModel):
    def __init__(self, config,dataset):
        super(TARNet, self).__init__(config,dataset)
        self.in_feature = self.dataset.size[1]
        self.alpha = self.config['alpha']
        self.bn = self.config['bn']
        self.repre_layer_sizes = self.config['repre_layer_sizes']
        self.pred_layer_sizes = self.config['pred_layer_sizes']


        self.repre_layers = nn.Sequential(*(([nn.BatchNorm1d(self.in_feature)] if self.bn else [])
                                             + get_linear_layers(self.in_feature,self.repre_layer_sizes,self.bn,nn.ReLU)))

        self.pred_layers_treated = nn.Sequential(*get_linear_layers(self.repre_layer_sizes[-1],
                                                                    self.pred_layer_sizes, False, nn.ReLU))

        self.pred_layers_treated.add_module('out1',nn.Linear(self.pred_layer_sizes[-1],1))
        self.pred_layers_control = nn.Sequential(*get_linear_layers(self.repre_layer_sizes[-1],
                                                                    self.pred_layer_sizes, False, nn.ReLU))
        self.pred_layers_control.add_module('out0', nn.Linear(self.pred_layer_sizes[-1],1))

        if self.loss_type == 'MSE':
            self.loss_fct = nn.MSELoss(reduction='none')
        elif self.loss_type == 'CE':
            self.loss_fct = nn.BCEWithLogitsLoss(reduction='none')
        else:
            raise NotImplementedError("Make sure 'loss_type' in ['MSE', 'CE']!")


    def forward(self, x, t):
        self.repre = self.repre_layers(x)
        treat_output = self.pred_layers_treated(self.repre)
        control_output = self.pred_layers_control(self.repre)
        return treat_output, control_output

    def get_repre(self, x, device):
        self.eval()
        with torch.no_grad():
            return self.repre_layers.to(device)(x.to(device))

    def calculate_loss(self, x,t,y,w):
        treat_output, control_output = self.forward(x, t)
        pred = torch.where(t == 1, treat_output, control_output)
        loss = torch.sum(self.loss_fct(pred,y) * w)
        return loss

    def predict(self, x,t):
        r"""Predict the scores between users and items.

        Args:
            interaction (Interaction): Interaction class of the batch.

        Returns:
            torch.Tensor: Predicted scores for given users and items, shape: [batch_size]
        """
        treat_output, control_output = self.forward(x, t)
        y = torch.where(t == 1, treat_output,control_output)
        if self.loss_type == 'MSE':
            return y
        else:
            return torch.sigmoid(y)

    def get_predict_yf(self,x,t):

        treat_output, control_output = self.forward(x, t)
        pred = torch.where(t == 1, treat_output, control_output)

        return pred.cpu().numpy()