import os
import torch
import torch.optim as optim
from torch.nn.utils.clip_grad import clip_grad_norm_

from time import time
from logging import getLogger
import torch.nn as nn
from causally.utils.utils import ensure_dir, get_local_time, early_stopping
from causally.trainer.AbstractTrainer import AbstractTrainer
class GTrainer(AbstractTrainer):

    def __init__(self, config, model):
        super(GTrainer, self).__init__(config, model)

        self.logger = getLogger()
        self.learner = config['optimizer']
        self.learning_rate = config['learning_rate']
        self.epochs = config['epochs']
        self.eval_step = min(config['eval_step'], self.epochs)
        self.stopping_step = config['stopping_step']
        self.clip_grad_norm = config['clip_grad_norm']
        self.valid_metric_bigger = config['valid_metric_bigger']
        self.test_batch_size = config['eval_batch_size']
        self.device = config['device']
        self.checkpoint_dir = config['checkpoint_dir']
        ensure_dir(self.checkpoint_dir)

        end = time()
        x = '-'.join(str(end).strip().split('.'))
        saved_model_file = '{}-{}-{}-{}.pth'.format(self.config['model'], self.config['dataset'], get_local_time(), x)
        self.saved_model_file = os.path.join(self.checkpoint_dir, saved_model_file)

        self.start_epoch = 0
        self.cur_step = 0
        self.best_valid_score = 100000000.
        self.best_valid_result = None
        self.train_loss_dict = dict()
        self.optimizer = self._build_optimizer()


    def to_device(self,x,t,y,w):
        return x.to(self.device),t.to(self.device),y.to(self.device),w.to(self.device)

    def _build_optimizer(self):

        if self.learner.lower() == 'adam':
            optimizer = optim.Adam(self.model.parameters(), lr=self.learning_rate)
        elif self.learner.lower() == 'sgd':
            optimizer = optim.SGD(self.model.parameters(), lr=self.learning_rate)
        elif self.learner.lower() == 'adagrad':
            optimizer = optim.Adagrad(self.model.parameters(), lr=self.learning_rate)
        elif self.learner.lower() == 'rmsprop':
            optimizer = optim.RMSprop(self.model.parameters(), lr=self.learning_rate)
        else:
            self.logger.warning('Received unrecognized optimizer, set default Adam optimizer')
            optimizer = optim.Adam(self.model.parameters(), lr=self.learning_rate)
        return optimizer

    @torch.no_grad()
    def _valid_epoch(self, val_data,loss_func=None):
        self.model.eval()
        n_samples = val_data.get_X_size()[0]
        loss_func = loss_func or self.model.calculate_loss
        total_loss = None
        for batch_data in val_data:
            x, t, y, w = self.to_device(batch_data[0], batch_data[1], batch_data[2], batch_data[3])
            losses = loss_func(x,t,y)
            total_loss = losses.item() if total_loss is None else total_loss + losses.item()
        total_loss = abs(total_loss)
        return total_loss / n_samples


    def _train_epoch(self, train_data, loss_func=None):

        self.model.train()
        n_samples = train_data.get_X_size()[0]
        loss_func = loss_func or self.model.calculate_loss
        total_loss = None
        for i,batch_data in enumerate(train_data):
            x, t, y, w = self.to_device(batch_data[0], batch_data[1], batch_data[2], batch_data[3])
            self.optimizer.zero_grad()
            losses = loss_func(x,t,y)
            if isinstance(losses, tuple):
                loss = sum(losses)
                loss_tuple = tuple(per_loss.item() for per_loss in losses)
                total_loss = loss_tuple if total_loss is None else tuple(map(sum, zip(total_loss, loss_tuple)))
            else:
                loss = losses
                total_loss = losses.item() if total_loss is None else total_loss + losses.item()

            self._check_nan(loss)
            loss.backward()
            if self.clip_grad_norm:
                clip_grad_norm_(self.model.parameters(), self.clip_grad_norm)
            self.optimizer.step()

        return total_loss / n_samples



    def _save_checkpoint(self, epoch):

        state = {
            'config': self.config,
            'epoch': epoch,
            'cur_step': self.cur_step,
            'best_valid_score': self.best_valid_score,
            'state_dict': self.model.state_dict(),
            'optimizer': self.optimizer.state_dict(),

        }
        torch.save(state, self.saved_model_file)

    def resume_checkpoint(self, resume_file):

        resume_file = str(resume_file)
        checkpoint = torch.load(resume_file,weights_only=False)
        self.start_epoch = checkpoint['epoch'] + 1
        self.cur_step = checkpoint['cur_step']
        self.best_valid_score = checkpoint['best_valid_score']

        # load architecture params from checkpoint
        if checkpoint['config']['model'].lower() != self.config['model'].lower():
            self.logger.warning('Architecture configuration given in config file is different from that of checkpoint. '
                                'This may yield an exception while state_dict is being loaded.')
        self.model.load_state_dict(checkpoint['state_dict'])

        # load optimizer state from checkpoint only when optimizer type is not changed
        self.optimizer.load_state_dict(checkpoint['optimizer'])
        message_output = 'Checkpoint loaded. Resume training from epoch {}'.format(self.start_epoch)
        self.logger.info(message_output)

    def _check_nan(self, loss):
        if torch.isnan(loss):
            raise ValueError('Training loss is nan')

    def _generate_train_loss_output(self, epoch_idx, s_time, e_time, losses):
        train_loss_output = 'epoch %d training [time: %.2fs, ' % (epoch_idx, e_time - s_time)
        if isinstance(losses, tuple):
            train_loss_output = ', '.join('train_loss%d: %.4f' % (idx + 1, loss) for idx, loss in enumerate(losses))
        else:
            train_loss_output += 'train loss: %.4f' % losses
        return train_loss_output + ']'

    def fit(self, train_data, valid_data=None, verbose=True, saved=True):


        if saved and self.start_epoch >= self.epochs:
            self._save_checkpoint(-1)

        for epoch_idx in range(self.start_epoch, self.epochs):
            # train
            training_start_time = time()
            train_loss = self._train_epoch(train_data)
            self.train_loss_dict[epoch_idx] = sum(train_loss) if isinstance(train_loss, tuple) else train_loss
            training_end_time = time()
            train_loss_output = \
                self._generate_train_loss_output(epoch_idx, training_start_time, training_end_time, train_loss)
            if verbose:
                self.logger.info(train_loss_output)

            # eval
            if self.eval_step <= 0 or not valid_data:
                if saved:
                    self._save_checkpoint(epoch_idx)
                    update_output = 'Saving current: %s' % self.saved_model_file
                    if verbose:
                        self.logger.info(update_output)
                continue
            if (epoch_idx + 1) % self.eval_step == 0:
                valid_start_time = time()
                valid_score = self._valid_epoch(valid_data)
                self.best_valid_score, self.cur_step, stop_flag, update_flag = early_stopping(
                    valid_score, self.best_valid_score, self.cur_step,
                    max_step=self.stopping_step, bigger=self.valid_metric_bigger)
                valid_end_time = time()
                valid_score_output = "epoch %d evaluating [time: %.2fs, valid_score: %f]" % \
                                     (epoch_idx, valid_end_time - valid_start_time, valid_score)
                # valid_result_output = 'valid result: \n' + dict2str(valid_result)
                if verbose:
                    self.logger.info(valid_score_output)
                    # self.logger.info(valid_result_output)
                if update_flag:
                    if saved:
                        self._save_checkpoint(epoch_idx)
                        update_output = 'Saving current best: %s' % self.saved_model_file
                        if verbose:
                            self.logger.info(update_output)
                    # self.best_valid_result = valid_result

                if stop_flag:
                    stop_output = 'Finished training, best eval result in epoch %d' % \
                                  (epoch_idx - self.cur_step * self.eval_step)
                    if verbose:
                        self.logger.info(stop_output)
                    break


        checkpoint_file = self.saved_model_file
        checkpoint = torch.load(checkpoint_file,weights_only=False)
        self.model.load_state_dict(checkpoint['state_dict'])
        message_output = 'Loading model structure and parameters from {}\n'.format(checkpoint_file)
        self.logger.info(message_output)

    @torch.no_grad()
    def generation(self, x):

        if self.config['covariate'] == 'gaussian':
            covariates = torch.randn(x.shape).to(self.device)
        elif self.config['covariate'] == 'uniform':
            covariates = torch.randn(x.shape).to(self.device)
            covariates.uniform_(-1.5,1.5)
        else:
            self.model.eval()
            covariates = self.model.generation(x)

        return covariates

    @torch.no_grad()
    def get_eta(self, x, t, y):
        if self.config['eta']:
            a, b = self.config['eta_a'], self.config['eta_b']
            if self.config['eta'] == 'gaussian':
                covariates = torch.normal(mean=a,std=b,size=(x.shape[0],1)).to(self.device).squeeze(-1)
            elif self.config['eta'] == 'uniform':
                covariates = torch.randn(size=(x.shape[0],1)).to(self.device).squeeze(-1)
                covariates.uniform_(a, b)
            else:
                raise 'No supported eta types!!!'
            covariates = self.model.get_noise_eta(x, t, y,covariates)
            return covariates
        else:
            self.model.eval()
            covariates = self.model.get_eta(x, t, y)
            if self.config['latent']:
                a, b = self.config['latent_a'], self.config['latent_b']
                if self.config['latent'] == 'gaussian':
                    perturbations = torch.normal(mean=a, std=b, size=x.shape).to(self.device)
                elif self.config['latent'] == 'uniform':
                    perturbations = torch.randn(size=x.shape).to(self.device)
                    perturbations.uniform_(a, b)
                else:
                    raise 'No supported eta types!!!'
                covariates = covariates + perturbations
            return covariates

    # @torch.no_grad()
    # def get_eta(self,x,t,y):
    #     if self.config['eta']:
    #         a,b = self.config['eta_a'],self.config['eta_b']
    #         if self.config['eta'] == 'gaussian':
    #             covariates = torch.normal(mean=a,std=b,size=x.shape).to(self.device)
    #         elif self.config['eta'] == 'uniform':
    #             covariates = torch.randn(x.shape).to(self.device)
    #             covariates.uniform_(a, b)
    #         else:
    #             raise 'No supported eta types!!!'
    #         return covariates
    #     else:
    #         self.model.eval()
    #         return self.model.get_eta(x,t,y)