from __future__ import annotations

from typing import List, Dict

from .llm_client import LLMClient
from .schemas import ScenarioDefinition, ExpertInput, ExpertOutput, ConflictReport
from . import prompts


class ProblemRefinerAgent:
    def __init__(self, client: LLMClient) -> None:
        self.client = client

    def refine(self, proposition: str) -> ScenarioDefinition:
        system = prompts.system_prefix("ProblemRefinement")
        user = prompts.problem_refinement_prompt(proposition)
        resp = self.client.chat([
            {"role": "system", "content": system},
            {"role": "user", "content": user},
        ], temperature=0.0)
        text = resp["choices"][0]["message"]["content"].strip()

        # Light parsing heuristic: attempt to split sections while keeping NL text.
        sections = {"premises": [], "constraints": [], "timescales": [], "uncertainties": [], "expert_plan": []}
        current = None
        for line in text.splitlines():
            lower = line.lower()
            if "premise" in lower and ":" in line:
                current = "premises"; continue
            if "constraint" in lower and ":" in line:
                current = "constraints"; continue
            if "timescale" in lower and ":" in line:
                current = "timescales"; continue
            if "uncertaint" in lower and ":" in line:
                current = "uncertainties"; continue
            if "expert plan" in lower and ":" in line:
                current = "expert_plan"; continue
            if current:
                if line.strip():
                    sections[current].append(line.strip(" -•\t"))

        return ScenarioDefinition(
            proposition=proposition,
            premises=sections["premises"] or ["Assumptions clarified in natural language above."],
            constraints=sections["constraints"] or ["Modern technology level maintained unless stated."],
            timescales=sections["timescales"] or ["Short-term", "Medium-term", "Long-term"],
            uncertainties=sections["uncertainties"] or ["Not specified; treat as open variables."],
            expert_plan=sections["expert_plan"] or [
                "Physics","ChemistryMaterials","BiologyEcology","Medicine","Sociology","Economics","PoliticsIR","EngineeringInfrastructure","EnvironmentalScience"
            ],
            refinement_raw=text,
        )


class DomainExpertAgent:
    def __init__(self, role: str, client: LLMClient) -> None:
        self.role = role
        self.client = client

    def run(self, expert_input: ExpertInput) -> ExpertOutput:
        scenario_text = _scenario_to_text(expert_input.scenario)
        system = prompts.system_prefix(self.role)
        user = prompts.expert_round_prompt(self.role, scenario_text, expert_input.shared_frame_summary, expert_input.round_index)
        resp = self.client.chat([
            {"role": "system", "content": system},
            {"role": "user", "content": user},
        ], temperature=0.0)
        text = resp["choices"][0]["message"]["content"].strip()

        # Non-strict parsing by headings
        sections = {"reasoning": [], "conclusions": [], "assumptions": [], "uncertainties": [], "dependency": []}
        current = None
        for line in text.splitlines():
            lower = line.lower()
            if "reasoning" in lower and ":" in line:
                current = "reasoning"; continue
            if "conclusion" in lower and ":" in line:
                current = "conclusions"; continue
            if "assumption" in lower and ":" in line:
                current = "assumptions"; continue
            if "uncertaint" in lower and ":" in line:
                current = "uncertainties"; continue
            if "dependency" in lower and ":" in line:
                current = "dependency"; continue
            if current and line.strip():
                sections[current].append(line.strip(" -•\t"))

        out = ExpertOutput(
            role=self.role,
            reasoning_steps=sections["reasoning"] or [text],
            conclusions=sections["conclusions"] or [],
            assumptions=sections["assumptions"] or [],
            uncertainties=sections["uncertainties"] or [],
            dependency_notes=sections["dependency"] or [],
            raw_text=text,
        )

        print(f' - Expert {self.role} Finished.')
        return out


class ConflictResolverAgent:
    def __init__(self, client: LLMClient) -> None:
        self.client = client

    def reconcile(self, expert_outputs: List[ExpertOutput]) -> ConflictReport:
        # Join expert outputs as natural language
        aggregate = []
        for out in expert_outputs:
            block = [
                f"Role: {out.role}",
                "Reasoning:",
                *out.reasoning_steps,
                "Conclusions:",
                *out.conclusions,
                "Assumptions:",
                *out.assumptions,
                "Uncertainties:",
                *out.uncertainties,
                "Dependencies:",
                *out.dependency_notes,
            ]
            aggregate.append("\n".join(block))

        system = prompts.system_prefix("ConflictResolver")
        user = prompts.conflict_detection_prompt(aggregate)
        resp = self.client.chat([
            {"role": "system", "content": system},
            {"role": "user", "content": user},
        ], temperature=0.0)
        text = resp["choices"][0]["message"]["content"].strip()

        # Heuristic parse
        consensus, branches, remaining, notes = [], {}, [], []
        current = None
        for line in text.splitlines():
            lower = line.lower()
            if "consensus" in lower and ":" in line:
                current = "consensus"; continue
            if "branch" in lower and ":" in line:
                current = "branches"; continue
            if "uncertaint" in lower and ":" in line:
                current = "remaining"; continue
            if "note" in lower and ":" in line:
                current = "notes"; continue
            if not line.strip():
                continue
            if current == "consensus":
                consensus.append(line.strip(" -•\t"))
            elif current == "branches":
                # Try parse 'condition -> description'
                if "->" in line:
                    cond, desc = line.split("->", 1)
                    branches[cond.strip(" -•\t:")] = desc.strip()
                else:
                    branches[line.strip(" -•\t")] = ""
            elif current == "remaining":
                remaining.append(line.strip(" -•\t"))
            elif current == "notes":
                notes.append(line.strip(" -•\t"))

        return ConflictReport(
            consensus_points=consensus,
            conditional_branches=branches,
            remaining_uncertainties=remaining,
            notes=notes,
            raw_text=text,
        )


class ReportGeneratorAgent:
    def __init__(self, client: LLMClient) -> None:
        self.client = client

    def generate(self, proposition: str, rounds_summaries: List[str]) -> str:
        system = prompts.system_prefix("ReportGenerator")
        user = prompts.report_generation_prompt(proposition, rounds_summaries)
        resp = self.client.chat([
            {"role": "system", "content": system},
            {"role": "user", "content": user},
        ], temperature=0.0)
        return resp["choices"][0]["message"]["content"].strip()


class DebateCritiqueAgent:
    """A lightweight debate-style Pro/Con synthesizer to inject contrastive signals per round.

    Produces a compact judge brief to be appended into the conflict resolver's shared summary,
    improving calibration and diversity while keeping call budget small (3 calls/round).
    """

    def __init__(self, client: LLMClient) -> None:
        self.client = client

    def synthesize(self, proposition: str, scenario: ScenarioDefinition, round_index: int) -> str:
        scenario_text = _scenario_to_text(scenario)
        transcripts = []
        for side in ["Pro", "Con"]:
            system = prompts.system_prefix(f"Debater-{side}")
            user = prompts.debate_arg_prompt(side, proposition, scenario_text, round_index)
            resp = self.client.chat([
                {"role": "system", "content": system},
                {"role": "user", "content": user},
            ], temperature=0.0)
            transcripts.append(f"[{side}]\n{resp['choices'][0]['message']['content'].strip()}")

        judge_system = prompts.system_prefix("DebateJudge")
        joined = "\n\n---\n\n".join(transcripts)
        judge_user = prompts.debate_judge_prompt(proposition, joined)
        judge = self.client.chat([
            {"role": "system", "content": judge_system},
            {"role": "user", "content": judge_user},
        ], temperature=0.0)
        brief = judge["choices"][0]["message"]["content"].strip()
        return brief


# RubricPolisherAgent removed per fairness policy


def _scenario_to_text(s: ScenarioDefinition) -> str:
    return (
        f"Proposition: {s.proposition}\n"
        f"Premises: \n- " + "\n- ".join(s.premises) + "\n"
        f"Constraints: \n- " + "\n- ".join(s.constraints) + "\n"
        f"Timescales: \n- " + "\n- ".join(s.timescales) + "\n"
        f"Key Uncertainties: \n- " + "\n- ".join(s.uncertainties) + "\n"
        f"Expert Plan: \n- " + "\n- ".join(s.expert_plan)
    )

