import os
from typing import List, Dict, Any, Optional

from openai import OpenAI


class LLMClient:
    """Light wrapper around OpenAI-compatible client for Volcengine Ark Doubao.

    Reads API Key from env var `API_KEY` by default. Allows overriding base_url and model.
    """

    def __init__(
        self,
        base_url: str = "https://ark.cn-beijing.volces.com/api/v3",
        api_key_env: str = "API_KEY",
        default_model: str = "doubao-seed-1-6-flash-250715",
    ) -> None:
        api_key = os.environ.get(api_key_env)
        if not api_key:
            raise RuntimeError(
                f"Missing API key in environment variable {api_key_env}."
            )
        self.client = OpenAI(base_url=base_url, api_key=api_key)
        self.default_model = default_model

    def chat(
        self,
        messages: List[Dict[str, Any]],
        model: Optional[str] = None,
        temperature: float = 0.2,
        max_tokens: Optional[int] = None,
    ) -> Dict[str, Any]:
        response = self.client.chat.completions.create(
            model=model or self.default_model,
            messages=messages,
            temperature=temperature,
            max_tokens=max_tokens,
        )
        # Normalize to Python dict
        if hasattr(response, "to_dict_recursive"):
            return response.to_dict_recursive()
        try:
            # OpenAI 1.x objects often have .model_dump
            return response.model_dump()
        except Exception:
            # Last resort: build minimal expected shape
            content = None
            try:
                content = response.choices[0].message.content  # type: ignore[attr-defined]
            except Exception:
                pass
            return {"choices": [{"message": {"content": content}}]}

