from typing import List


def system_prefix(role: str) -> str:
    return (
        "You are a rigorous domain expert AI optimized for cross-disciplinary counterfactual reasoning. "
        f"Role: {role}. "
        "Think step by step with explicit causal reasoning and cross-domain links. "
        "Prefer probabilistic, non-deterministic phrasing with order-of-magnitude ranges. "
        "Model substitution and adaptation when projecting impacts; avoid one-way collapse narratives. "
        "Explicitly audit feasibility (resource/tech/policy constraints) and trace every claim to assumptions. "
        "Always structure outputs with clear headings and bullet points aligned to: coverage, causality, feasibility, uncertainty/adaptation, scenarios/timeline, traceability/consistency. "
        "Use compact tables when helpful. Outputs MUST be in English."
    )


def problem_refinement_prompt(proposition: str) -> str:
    return (
        "Task: Convert the vague proposition into an actionable scenario definition. Mainly consider the impact to the world.\n"
        f"Proposition: {proposition}\n\n"
        "Deliver: \n"
        "1) Premises (key setup details; resolve ambiguities reasonably if needed)\n"
        "2) Constraints (hard assumptions to hold constant)\n"
        "3) Timescales (short/medium/long term)\n"
        "4) Key Uncertainties (variables that may branch outcomes)\n"
        "5) Expert Plan (list of domains to involve)\n"
        "Also list 2-3 scenario variants (e.g., fast vs. gradual change) and 3-5 measurable indicators to track.\n"
        "Provide a structured but natural language answer."
    )


def expert_round_prompt(role: str, scenario_text: str, shared_summary: str | None, round_index: int) -> str:
    intro = (
        "Round 1: Work independently without relying on other domains.\n"
        if round_index == 1
        else "Round 2+: Use the shared frame summary for alignment; resolve conflicts and refine conditional scenarios with explicit numbers/ranges.\n"
    )
    shared = f"\nShared Frame Summary:\n{shared_summary}\n" if shared_summary else "\n"
    return (
        f"{intro}Domain: {role}.\n"
        f"Scenario:\n{scenario_text}{shared}\n"
        "Produce: \n"
        "- Reasoning Steps (explicit causal links; cite physics/resources/biology/society/economy as relevant)\n"
        "- One-sentence Verdict (sharp, testable)\n"
        "- Conclusions (concise; include 1-2 quantitative ranges)\n"
        "- Feasibility Audit (resource, engineering, policy constraints; show bottlenecks)\n"
        "- Feasibility Table (rows: constraint/capacity; cols: estimate, unit, bottleneck, mitigation)\n"
        "- Assumptions (explicit)\n"
        "- Uncertainties (drivers; include substitution/adaptation levers)\n"
        "- Calibration Ranges: for key quantities provide 50% / 80% / 95% intervals\n"
        "- Dependency Notes (which other domains drive your conclusions)\n"
        "- Minimal-change Variant (minimal extra assumptions to retain conclusions)\n"
        "- Scenarios & Timeline (Short/Medium/Long; attach indicative probabilities %)\n"
        "- Observable Indicators (3-5) with thresholds and how they flip branches\n"
        "- Assumption->Claim Trace Table (2 columns: assumption_id -> supported_claim_id; keep entries short)\n"
        "- Cross-domain Causal Integration Matrix (rows: mechanisms; columns: domains; cells: (+/−, strength 1-3), list key edges)\n"
    )


def conflict_detection_prompt(aggregate_texts: List[str]) -> str:
    joined = "\n\n---\n\n".join(aggregate_texts)
    return (
        "You are a conflict detection and reconciliation AI.\n"
        "Compare the following domain outputs, identify conflicts, categorize them (hard, soft, granularity), and synthesize a unified multi-scenario frame.\n\n"
        f"{joined}\n\n"
        "Deliver: \n"
        "- Consensus Points (note confidence level) — list first\n"
        "- Conditional Branches (condition -> description; domains driving; attach rough probabilities %)\n"
        "- Decision Rules (observable indicators with thresholds -> which branch)\n"
        "- Remaining Uncertainties (include measurable indicators)\n"
        "- Brief Notes (how conflicts were treated; prioritize physics > biology survival > basic resources > social > economy)\n"
        "Use compact bullets and natural language."
    )


def report_generation_prompt(proposition: str, rounds_summaries: List[str]) -> str:
    body = "\n\n".join(rounds_summaries)
    return (
        "You are a report generation AI that writes a readable, well-structured multi-agent reasoning report.\n"
        f"Proposition: {proposition}\n\n"
        f"Inputs from rounds:\n{body}\n\n"
        "Deliver: \n"
        "- Rubric-aligned Summary (5 bullets): Rigor/Trace; Integration/Causality; Feasibility/Minimality; Uncertainty/Adaptation; Decisionability. Keep quant and indicators upfront.\n"
        "- Executive Verdict (single-sentence, sharp; include feasibility+minimal-change statement)\n"
        "- Core conclusions and uncertainty analysis (include adaptation/substitution and 1-2 quantitative ranges)\n"
        "- Traceability Summary: numbered assumptions and claims, plus an Assumption->Claim Trace Table (compact)\n"
        "- Cross-domain Causal Integration Matrix: mechanisms vs. domains with (+/−, strength), list top 6-10 edges\n"
        "- Feasibility Table (constraint/capacity with estimates, unit, bottleneck, mitigation)\n"
        "- Calibration Ranges table (key quantities with 50% / 80% / 95% intervals)\n"
        "- Alignment Summary (consensus first; retained branches with conditions and rough probabilities %)\n"
        "- Decision Table (observable indicators & thresholds -> scenario branch selection)\n"
        "- Causal map (nodes and edges in bullet form; cross-domain links)\n"
        "- Multi-scenario analysis (Scenario 1..N; drivers, pros/cons, indicative probabilities, measurable indicators)\n"
        "- Timeline of events (short/medium/long; concise table)\n"
        "- Consistency checks (how conflicts were resolved; residual disagreements)\n"
        "Prefer natural language; include minimal JSON/tables only if helpful."
    )


# Removed rubric polisher to avoid leakage or post-hoc optimization


# Debate-style prompts to integrate Pro/Con synthesis within the multi-agent pipeline
def debate_arg_prompt(side: str, proposition: str, scenario_text: str, round_index: int) -> str:
    return (
        f"Debate role: {side}.\n"
        f"Round {round_index}.\n"
        f"Proposition: {proposition}\n\n"
        f"Scenario (for context):\n{scenario_text}\n\n"
        "Make a concise argument covering physics/resources/biology/society/economy.\n"
        "Emphasize uncertainties, substitution/adaptation, and decision indicators with thresholds. Use English."
    )


def debate_judge_prompt(proposition: str, transcripts_joined: str) -> str:
    return (
        "As a judge, synthesize the Pro/Con arguments into a balanced cross-domain brief.\n"
        f"Proposition: {proposition}\n\n"
        f"Debate Transcript:\n{transcripts_joined}\n\n"
        "Deliver: 5-8 bullet conclusions (with 1-2 quantitative ranges), key uncertainties/adaptation levers,\n"
        "2-4 conditional branches with rough probabilities, and 3-5 decision rules (indicator -> branch)."
    )

