from __future__ import annotations

from typing import Dict, Any, Tuple
from pathlib import Path

from src.a4s.llm_client import LLMClient
from src.a4s import prompts


def _arg_prompt(side: str, proposition: str) -> str:
    return (
        f"Debate role: {side}.\n"
        f"Proposition: {proposition}\n"
        "Make a concise argument covering physics/resources/biology/society/economy.\n"
        "Emphasize uncertainties and possible adaptations. Use English."
    )


def run_debate(proposition: str, client: LLMClient, out_dir: Path, rounds: int = 3) -> Tuple[str, Dict[str, Any]]:
    """Two-agent debate with a judge synthesis. Total calls ~ 2*rounds + 1 <= 25 by default."""
    out_dir.mkdir(parents=True, exist_ok=True)

    transcripts = []
    for r in range(1, rounds + 1):
        for side in ["Pro", "Con"]:
            system = prompts.system_prefix(f"Debater-{side}")
            user = _arg_prompt(side, proposition) + f"\nRound {r}."
            resp = client.chat([
                {"role": "system", "content": system},
                {"role": "user", "content": user},
            ])
            text = resp["choices"][0]["message"]["content"].strip()
            transcripts.append(f"[{side} R{r}]\n{text}")

    # Judge synthesis
    system = prompts.system_prefix("DebateJudge")
    joined = "\n\n---\n\n".join(transcripts)
    user = (
        "As a judge, synthesize the debate into a balanced cross-domain report.\n"
        f"Proposition: {proposition}\n\n"
        f"Debate Transcript:\n{joined}\n\n"
        "Deliver: conclusions with uncertainty, branches, timeline, and adaptation notes."
    )
    resp = client.chat([
        {"role": "system", "content": system},
        {"role": "user", "content": user},
    ])
    report = resp["choices"][0]["message"]["content"].strip()

    (out_dir / "report.md").write_text(report, encoding="utf-8")
    structured = {"proposition": proposition, "rounds": [f"debate rounds={rounds}"], "model": "debate"}
    (out_dir / "structured.json").write_text(__import__("json").dumps(structured, ensure_ascii=False, indent=2), encoding="utf-8")
    (out_dir / "transcript.txt").write_text("\n\n".join(transcripts), encoding="utf-8")

    return report, structured


