from __future__ import annotations

from dataclasses import dataclass
from typing import List, Dict, Any, Tuple
from pathlib import Path

from src.a4s.llm_client import LLMClient
from src.a4s import prompts


@dataclass
class Node:
    prompt: str
    text: str | None = None
    score: float | None = None


def _score_answer(proposition: str, answer: str, client: LLMClient) -> float:
    system = prompts.system_prefix("CriticScorer")
    user = (
        "Rate the following answer for cross-disciplinary plausibility, clarity, and explicit uncertainty handling on a 0-10 scale.\n"
        f"Proposition: {proposition}\n\nAnswer:\n{answer}\n\n"
        "Return only a number between 0 and 10."
    )
    resp = client.chat([
        {"role": "system", "content": system},
        {"role": "user", "content": user},
    ], temperature=0.0, max_tokens=64)
    txt = resp["choices"][0]["message"]["content"].strip()
    try:
        return float(txt.split()[0])
    except Exception:
        return 5.0


def run_tree_search(
    proposition: str,
    client: LLMClient,
    out_dir: Path,
    breadth: int = 2,
    depth: int = 2,
) -> Tuple[str, Dict[str, Any]]:
    out_dir.mkdir(parents=True, exist_ok=True)

    root_user = (
        "Draft an initial cross-domain analysis for the proposition. Include uncertainties and adaptation."
        f"\nProposition: {proposition}"
    )
    root = Node(prompt=root_user)

    drafts: List[Node] = []
    for _ in range(depth):
        candidates: List[Node] = []
        seeds = drafts if drafts else [root]
        for parent in seeds:
            for b in range(breadth):
                system = prompts.system_prefix("TreeSearchDraft")
                user = parent.prompt + f"\nVariant #{b+1}: explore different plausible assumptions and branches."
                resp = client.chat([
                    {"role": "system", "content": system},
                    {"role": "user", "content": user},
                ])
                text = resp["choices"][0]["message"]["content"].strip()
                candidates.append(Node(prompt=user, text=text))
        for node in candidates:
            node.score = _score_answer(proposition, node.text or "", client)
        candidates.sort(key=lambda n: (n.score or 0.0), reverse=True)
        drafts = candidates[:breadth]

    best = max(drafts, key=lambda n: (n.score or 0.0))
    (out_dir / "report.md").write_text(best.text or "", encoding="utf-8")
    structured = {
        "proposition": proposition,
        "rounds": [f"tree_search depth={depth} breadth={breadth}"],
        "model": "tree-search",
        "score": best.score,
    }
    (out_dir / "structured.json").write_text(__import__("json").dumps(structured, ensure_ascii=False, indent=2), encoding="utf-8")
    return best.text or "", structured


