#!/usr/bin/env python3
import argparse
import json
import os
from pathlib import Path
from datetime import datetime

from src.a4s.schemas import OrchestratorConfig
from src.a4s.orchestrator import Orchestrator

try:
    from dotenv import load_dotenv, find_dotenv
    _DOTENV_AVAILABLE = True
except Exception:
    _DOTENV_AVAILABLE = False


def load_config_from_json(path: Path) -> OrchestratorConfig:
    if not path.exists() or path.stat().st_size == 0:
        return OrchestratorConfig()
    with path.open("r", encoding="utf-8") as f:
        data = json.load(f)
    return OrchestratorConfig(
        rounds=data.get("rounds", 2),
        expert_roles=data.get("expert_roles") or OrchestratorConfig().expert_roles,
        dependency_map=data.get("dependency_map") or OrchestratorConfig().dependency_map,
        model_id=data.get("model_id", OrchestratorConfig().model_id),
    )


def main() -> None:
    if _DOTENV_AVAILABLE:
        load_dotenv(find_dotenv(), override=False)

    parser = argparse.ArgumentParser(description="Agents4Sci multi-agent hypothetical scenario reasoning")
    parser.add_argument("-q", "--query", type=str, help="Hypothetical proposition to analyze")
    parser.add_argument("-c", "--config", type=str, default="config.json", help="Path to config JSON")
    parser.add_argument("-o", "--outdir", type=str, default="outputs", help="Directory to save outputs")
    args = parser.parse_args()

    proposition = args.query or input("请输入假设命题 (e.g., 如果地球的重力突然变成10倍，会发生什么？): ").strip()
    cfg = load_config_from_json(Path(args.config))

    out_root = Path(args.outdir)
    timestamp = datetime.now().strftime("%Y%m%d-%H%M%S")
    out_dir = out_root / timestamp
    out_dir.mkdir(parents=True, exist_ok=True)

    orch = Orchestrator(cfg)
    final_report, structured = orch.run(proposition, out_dir=out_dir)

    (out_dir / "report.md").write_text(final_report, encoding="utf-8")
    (out_dir / "structured.json").write_text(json.dumps(structured, ensure_ascii=False, indent=2), encoding="utf-8")

    print(f"Saved report to {out_dir}/report.md")


if __name__ == "__main__":
    main()
    # conda run -n a4s python run.py -q 'What if 苏联赢得了冷战？?' -c config/config.json -o outputs/test


