from __future__ import annotations

from concurrent.futures import ThreadPoolExecutor, as_completed
from datetime import datetime
from typing import List, Dict, Tuple, Optional
from pathlib import Path

from .llm_client import LLMClient
from .schemas import (
    ScenarioDefinition,
    ExpertInput,
    ExpertOutput,
    ConflictReport,
    OrchestratorConfig,
)
from .agents import (
    ProblemRefinerAgent,
    DomainExpertAgent,
    ConflictResolverAgent,
    ReportGeneratorAgent,
    DebateCritiqueAgent,
)


class Orchestrator:
    def __init__(self, config: OrchestratorConfig) -> None:
        self.config = config
        self.client = LLMClient(default_model=config.model_id)
        self.refiner = ProblemRefinerAgent(self.client)
        self.conflict_resolver = ConflictResolverAgent(self.client)
        self.reporter = ReportGeneratorAgent(self.client)
        self.debate = DebateCritiqueAgent(self.client)

    def run(self, proposition: str, out_dir: Optional[Path] = None) -> Tuple[str, Dict[str, object]]:
        # 1) Problem refinement
        print(f'Problem Refinement Started.')
        scenario = self.refiner.refine(proposition)

        # Prepare logging dirs if provided
        logs_root: Optional[Path] = None
        if out_dir is not None:
            logs_root = Path(out_dir) / "logs"
            logs_root.mkdir(parents=True, exist_ok=True)
            # Save scenario refinement raw
            (logs_root / "scenario.md").write_text(scenario.refinement_raw or "", encoding="utf-8")

        # Allow scenario.expert_plan to override configured roles
        roles = scenario.expert_plan or self.config.expert_roles

        rounds_summaries: List[str] = []
        shared_summary: str | None = None

        total_rounds = max(self.config.rounds, 2)
        for round_index in range(1, total_rounds + 1):
            print(f'Round {round_index} Started.')
            # 2) Parallel domain expert runs
            experts = [DomainExpertAgent(role, self.client) for role in roles]
            expert_input = ExpertInput(scenario=scenario, shared_frame_summary=shared_summary, round_index=round_index)
            outputs: List[ExpertOutput] = []
            with ThreadPoolExecutor(max_workers=min(len(experts), 12)) as pool:
                futures = {pool.submit(exp.run, expert_input): exp.role for exp in experts}
                for fut in as_completed(futures):
                    outputs.append(fut.result())

            # Write expert logs
            if logs_root is not None:
                round_dir = logs_root / f"round_{round_index}"
                experts_dir = round_dir / "experts"
                experts_dir.mkdir(parents=True, exist_ok=True)
                for out in outputs:
                    # Save raw text and parsed JSON
                    (experts_dir / f"{out.role}.md").write_text(out.raw_text or "", encoding="utf-8")
                    import json as _json
                    (experts_dir / f"{out.role}.json").write_text(
                        _json.dumps({
                            "role": out.role,
                            "reasoning_steps": out.reasoning_steps,
                            "conclusions": out.conclusions,
                            "assumptions": out.assumptions,
                            "uncertainties": out.uncertainties,
                            "dependency_notes": out.dependency_notes,
                        }, ensure_ascii=False, indent=2),
                        encoding="utf-8",
                    )

            # 3) Conflict detection and reconciliation
            print(f'Conflict Detection and Reconciliation Started.')
            # 3a) Inject a compact Pro/Con synthesis to enrich contrast before reconciliation
            try:
                debate_brief = self.debate.synthesize(proposition, scenario, round_index)
            except Exception:
                debate_brief = ""
            conflict: ConflictReport = self.conflict_resolver.reconcile(outputs)
            shared_summary = self._render_shared_summary(conflict)
            if debate_brief:
                shared_summary = shared_summary + "\n\nDebate Brief (contrastive synthesis):\n" + debate_brief

            # Write conflict and shared frame logs
            if logs_root is not None:
                round_dir = logs_root / f"round_{round_index}"
                round_dir.mkdir(parents=True, exist_ok=True)
                (round_dir / "conflict.md").write_text(conflict.raw_text or "", encoding="utf-8")
                (round_dir / "shared_frame.md").write_text(shared_summary, encoding="utf-8")
                if debate_brief:
                    (round_dir / "debate_brief.md").write_text(debate_brief, encoding="utf-8")

            # Build a concise round summary for final reporting
            round_summary = self._render_round_summary(round_index, outputs, conflict)
            rounds_summaries.append(round_summary)

            if logs_root is not None:
                round_dir = logs_root / f"round_{round_index}"
                (round_dir / "summary.md").write_text(round_summary, encoding="utf-8")

        # 4) Final report
        print(f'Final Report Generation Started.')
        final_report = self.reporter.generate(proposition, rounds_summaries)

        # Minimal structured artifact for downstream use
        structured = {
            "proposition": proposition,
            "scenario": {
                "premises": scenario.premises,
                "constraints": scenario.constraints,
                "timescales": scenario.timescales,
                "uncertainties": scenario.uncertainties,
                "expert_plan": roles,
            },
            "rounds": rounds_summaries,
        }
        return final_report, structured

    @staticmethod
    def _render_shared_summary(conflict: ConflictReport) -> str:
        parts = [
            "Consensus Points:",
            *[f"- {c}" for c in conflict.consensus_points],
            "Conditional Branches:",
            *[f"- {k} -> {v}" for k, v in conflict.conditional_branches.items()],
            "Remaining Uncertainties:",
            *[f"- {u}" for u in conflict.remaining_uncertainties],
            "Notes:",
            *[f"- {n}" for n in conflict.notes],
        ]
        return "\n".join(parts)

    @staticmethod
    def _render_round_summary(round_index: int, outputs: List[ExpertOutput], conflict: ConflictReport) -> str:
        parts: List[str] = [f"Round {round_index} Summary"]
        for out in outputs:
            parts.append(f"Domain: {out.role}")
            if out.conclusions:
                parts.append("Conclusions:")
                parts.extend([f"- {c}" for c in out.conclusions])
            if out.uncertainties:
                parts.append("Uncertainties:")
                parts.extend([f"- {u}" for u in out.uncertainties])
        parts.append("\nReconciled Frame:")
        parts.append(Orchestrator._render_shared_summary(conflict))
        return "\n".join(parts)

