from __future__ import annotations

from typing import Dict, Any, Tuple
from pathlib import Path

from src.a4s.llm_client import LLMClient
from src.a4s import prompts


def run_single_agent(proposition: str, client: LLMClient, out_dir: Path) -> Tuple[str, Dict[str, Any]]:
    """Single-agent baseline: minimal zero-shot one-pass analysis.

    Returns (report_markdown, structured_dict)
    """
    system = prompts.system_prefix("SingleAgent")
    user = (
        "You are a single expert tasked to analyze a hypothetical proposition across multiple domains.\n"
        "Follow these:",
    )
    user = (
        "You are a single expert tasked to analyze a hypothetical proposition across multiple domains.\n"
        f"Proposition: {proposition}\n\n"
        "Keep the answer in English, compact but complete."
    )

    resp = client.chat([
        {"role": "system", "content": system},
        {"role": "user", "content": user},
    ])
    text = resp["choices"][0]["message"]["content"].strip()

    # Save outputs
    out_dir.mkdir(parents=True, exist_ok=True)
    (out_dir / "report.md").write_text(text, encoding="utf-8")
    structured = {"proposition": proposition, "rounds": ["single"], "model": "single-agent"}
    (out_dir / "structured.json").write_text(__import__("json").dumps(structured, ensure_ascii=False, indent=2), encoding="utf-8")

    return text, structured


