"""
Copyright 2022 Div Garg. All rights reserved.

Example training code for IQ-Learn which minimially modifies `train_rl.py`.
"""

import datetime
import os
import random
import time
from collections import deque
from itertools import count
import types

import hydra
import numpy as np
import torch
import torch.nn.functional as F
import wandb
from omegaconf import DictConfig, OmegaConf
from tensorboardX import SummaryWriter

from wrappers.atari_wrapper import LazyFrames
from make_envs import make_env
from dataset.memory import Memory
from agent import make_agent
from utils.utils import eval_mode, average_dicts, get_concat_samples, evaluate, soft_update, hard_update, get_args, set_up_log_dirs, logging
# from utils.logger import Logger
from utils.wandb_logger import wandb_logger as Logger
from iq import iq_loss

import uuid

torch.set_num_threads(2)




@hydra.main(config_path="conf", config_name="config_reward_gen")
def main(cfg: DictConfig):
    args = get_args(cfg)
    # set args.pretrain to None, because now we are training from scratch
    args.pretrain = None
    config_dict = OmegaConf.to_container(args, resolve=True)

    # # create a unique prefix for the run
    # unique_id = uuid.uuid4().hex[:8]  # Shortened version of UUID
    # run_id = f"{args.seed}_train_iq_{unique_id}"


    logger = Logger(args)
    logdirs = set_up_log_dirs(args, logger.prefix)
    log_dir, wandb_dir, agent_save_dir, agent_best_dir, reward_save_dir, video_save_dir = logdirs
    logger._create_wandb(log_dir=wandb_dir)

    # set seeds
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)

    device = torch.device(args.device)
    if device.type == 'cuda' and torch.cuda.is_available() and args.cuda_deterministic:
        torch.backends.cudnn.benchmark = False
        torch.backends.cudnn.deterministic = True

    env_args = args.env
    env = make_env(args)
    eval_env = make_env(args)

    # Seed envs
    env.seed(args.seed)
    eval_env.seed(args.seed + 10)
    # set action space seed
    if hasattr(env.action_space, 'seed'):
        env.action_space.seed(args.seed)
        eval_env.action_space.seed(args.seed + 10)

    REPLAY_MEMORY = int(env_args.replay_mem)
    INITIAL_MEMORY = int(env_args.initial_mem)
    EPISODE_STEPS = int(env_args.eps_steps)
    EPISODE_WINDOW = int(env_args.eps_window)
    LEARN_STEPS = int(env_args.learn_steps)
    INITIAL_STATES = 128  # Num initial states to use to calculate value of initial state distribution s_0

    agent = make_agent(env.observation_space, env.action_space, args, load_agent_path=args.pretrain)

    # Load expert data
    expert_memory_replay = Memory(REPLAY_MEMORY//2, args.seed)
    expert_memory_replay.load(hydra.utils.to_absolute_path(f'./iq_learn/expert_dataset/iq_learn/experts/{args.env.demo}'),
                              num_trajs=args.expert.demos,
                              sample_freq=args.expert.subsample_freq,
                              seed=args.seed + 42)
    logging(f'--> Expert memory size: {expert_memory_replay.size()}')

    online_memory_replay = Memory(REPLAY_MEMORY//2, args.seed+1)


    # logdirs = set_up_log_dirs(args, run_id)
    # log_dir, wandb_dir, agent_save_dir, agent_best_dir, reward_save_dir, video_save_dir = logdirs

    # writer = SummaryWriter(log_dir=wandb_dir)
    # logger = Logger(wandb_dir,
    #                 log_frequency=args.log_interval,
    #                 writer=writer,
    #                 save_tb=True,
    #                 agent=args.agent.name)

    steps = 0
    if args.env.name.find('MiniGrid') >= 0:
        flag_gw = True
    else:
        flag_gw = False

    # track mean reward and scores
    scores_window = deque(maxlen=EPISODE_WINDOW)  # last N scores
    rewards_window = deque(maxlen=EPISODE_WINDOW)  # last N rewards
    if flag_gw:
        found_goal_window = deque(maxlen=EPISODE_WINDOW)  # last N success rate
    best_eval_returns = -np.inf

    learn_steps = 0
    begin_learn = False
    episode_reward = 0

    # Sample initial states from env
    state_0 = [env.reset()] * INITIAL_STATES
    if isinstance(state_0[0], LazyFrames):
        state_0 = np.array(state_0) / 255.0
    state_0 = torch.FloatTensor(np.array(state_0)).to(args.device)

    for epoch in count():
        state = env.reset()

        episode_reward = 0
        done = False

        start_time = time.time()
        for episode_step in range(EPISODE_STEPS):

            if steps < args.num_seed_steps:
                # Seed replay buffer with random actions
                action = env.action_space.sample()
            else:
                with eval_mode(agent):
                    action = agent.choose_action(state, sample=True)
            next_state, reward, done, _ = env.step(action)
            episode_reward += reward
            steps += 1
            if flag_gw and done:
                if reward > 0:
                    found_goal = 1
                else:
                    found_goal = 0
                found_goal_window.append(found_goal)
                    
            if learn_steps % args.env.eval_interval == 0:
                eval_returns, eval_timesteps, eval_found_goals = evaluate(agent, 
                                                                          eval_env, 
                                                                          args,
                                                                          logger=logger,
                                                                          num_episodes=args.eval.eps, 
                                                                          flag_gw=flag_gw,
                                                                          video_save_dir=video_save_dir,
                                                                          learn_steps=learn_steps)
                learn_steps += 1  # To prevent repeated eval at timestep 0
                # print('EVAL\tEp {}\tAverage reward: {:.2f}\t'.format(epoch, returns))

                logging('EVAL\tEp {}\tAverage reward: {:.2f}\t'.format(epoch, np.mean(eval_returns)))

                # if np.mean(eval_returns) > best_eval_returns:
                #     # Store best eval returns
                #     best_eval_returns = np.mean(eval_returns)
                #     wandb.run.summary["best_returns"] = best_eval_returns
                #     save(agent, epoch, args, output_dir=agent_best_dir)

            # only store done true when episode finishes without hitting timelimit (allow infinite bootstrap)
            done_no_lim = done
            if str(env.__class__.__name__).find('TimeLimit') >= 0 and episode_step + 1 == env._max_episode_steps:
                done_no_lim = 0
            online_memory_replay.add((state, next_state, action, reward, done_no_lim))

            if online_memory_replay.size() > INITIAL_MEMORY:
                # Start learning
                if begin_learn is False:
                    logging('Learn begins!')
                    begin_learn = True

                learn_steps += 1
                if learn_steps == LEARN_STEPS:
                    logging('Finished!')
                    wandb.finish()
                    return

                ######
                # IQ-Learn Modification
                agent.iq_update = types.MethodType(iq_update, agent)
                agent.iq_update_critic = types.MethodType(iq_update_critic, agent)
                losses = agent.iq_update(online_memory_replay,
                                         expert_memory_replay, logger, learn_steps)
                ######

                if learn_steps % args.log_interval == 0:
                    # for key, loss in losses.items():
                    #     writer.add_scalar(key, loss, global_step=learn_steps)
                    logging('TRAIN\tEp {}\tAverage reward: {:.2f}\t'.format(epoch, np.mean(rewards_window)))

            if done:
                break
            state = next_state

        rewards_window.append(episode_reward)
        log_dict = {}
        log_dict['train/episode'] = epoch
        log_dict['train/episode_reward'] = episode_reward
        log_dict['train/duration'] = time.time() - start_time
        # logger.log('train/episode', epoch, learn_steps)
        # logger.log('train/episode_reward', episode_reward, learn_steps)
        # logger.log('train/duration', time.time() - start_time, learn_steps)
        if flag_gw:
            log_dict['train/episode_found_goal'] = found_goal
            # logger.log('train/episode_found_goal', found_goal, learn_steps)
        
        logger.wandb_log(log_dict, learn_steps)
        # logger.dump(learn_steps, save=begin_learn)
        # print('TRAIN\tEp {}\tAverage reward: {:.2f}\t'.format(epoch, np.mean(rewards_window)))
        save(agent, epoch, args, output_dir=agent_save_dir)


def save(agent, epoch, args, output_dir='results'):
    if epoch % args.save_interval == 0:
        if args.method.type == "sqil":
            name = f'sqil_{args.env.name}'
        else:
            name = f'iq_{args.env.name}'

        if not os.path.exists(output_dir):
            os.mkdir(output_dir)
        agent.save(f'{output_dir}/{args.agent.name}_{name}_{epoch}')
        logging(f'Saved model at {output_dir}/{args.agent.name}_{name}_{epoch}')


# Minimal IQ-Learn objective
def iq_learn_update(self, policy_batch, expert_batch, logger, step):
    args = self.args
    policy_obs, policy_next_obs, policy_action, policy_reward, policy_done = policy_batch
    expert_obs, expert_next_obs, expert_action, expert_reward, expert_done = expert_batch

    if args.only_expert_states:
        expert_batch = expert_obs, expert_next_obs, policy_action, expert_reward, expert_done

    obs, next_obs, action, reward, done, is_expert = get_concat_samples(
        policy_batch, expert_batch, args)

    loss_dict = {}

    ######
    # IQ-Learn minimal implementation with X^2 divergence (~15 lines)
    # Calculate 1st term of loss: -E_(ρ_expert)[Q(s, a) - γV(s')]
    current_Q = self.critic(obs, action)
    y = (1 - done) * self.gamma * self.getV(next_obs)
    if args.train.use_target:
        with torch.no_grad():
            y = (1 - done) * self.gamma * self.get_targetV(next_obs)

    reward = (current_Q - y)[is_expert]
    loss = -(reward).mean()

    # 2nd term for our loss (use expert and policy states): E_(ρ)[Q(s,a) - γV(s')]
    value_loss = (self.getV(obs) - y).mean()
    loss += value_loss

    # Use χ2 divergence (adds a extra term to the loss)
    chi2_loss = 1/(4 * args.method.alpha) * (reward**2).mean()
    loss += chi2_loss
    ######

    self.critic_optimizer.zero_grad()
    loss.backward()
    self.critic_optimizer.step()
    return loss


NODE_TO_ONE_HOT = {
    # Empty square
    (1, 0, 0, 0): [0, 0, 0], # black
    # Wall
    (0, 1, 0, 0): [100, 100, 100], # gray
    # Goal
    (0, 0, 1, 0): [0, 255, 0],
    # Agent
    (0, 0, 0, 1): [255, 0, 0],
}


def grid_img_gen(grids_th, name):
    # grid is a numpy array of shape(4, 19, 19)
    # firstly revert it into color image
    grids = grids_th.cpu().numpy()
    grids = np.transpose(grids, (1, 2, 0))
    img = []
    for i in range(grids.shape[0]):
        for j in range(grids.shape[1]):
            grid = grids[i, j]
            img.append(NODE_TO_ONE_HOT[tuple(grid.astype(int))])
    # to make the image more clear, we need to resize the image
    img = np.array(img).astype(np.uint8)
    img = img.reshape(19, 19, 3)
    from PIL import Image
    img = Image.fromarray(img)
    img_resized = img.resize((190, 190), Image.NEAREST)
    img_resized.save(f'~/projects/goal_prox_il/iq_learn/debug_img/grid_{name}.jpg')

def iq_update_critic(self, policy_batch, expert_batch, logger, step):
    args = self.args
    policy_obs, policy_next_obs, policy_action, policy_reward, policy_done = policy_batch
    expert_obs, expert_next_obs, expert_action, expert_reward, expert_done = expert_batch

    if args.if_debug:
        # generate image for debugging. of policy_obs and expert_obs
        grid_policy = policy_obs[0]
        grid_img_gen(grid_policy, 'policy')
        grid_expert = expert_obs[0]
        grid_img_gen(grid_expert, 'expert')



    if args.only_expert_states:
        # Use policy actions instead of experts actions for IL with only observations
        expert_batch = expert_obs, expert_next_obs, policy_action, expert_reward, expert_done

    batch = get_concat_samples(policy_batch, expert_batch, args)
    obs, next_obs, action = batch[0:3]

    agent = self
    current_V = self.getV(obs)
    if args.train.use_target:
        with torch.no_grad():
            next_V = self.get_targetV(next_obs)
    else:
        next_V = self.getV(next_obs)

    if "DoubleQ" in self.args.q_net._target_:
        current_Q1, current_Q2 = self.critic(obs, action, both=True)
        q1_loss, loss_dict1 = iq_loss(agent, current_Q1, current_V, next_V, batch)
        q2_loss, loss_dict2 = iq_loss(agent, current_Q2, current_V, next_V, batch)
        critic_loss = 1/2 * (q1_loss + q2_loss)
        # merge loss dicts
        loss_dict = average_dicts(loss_dict1, loss_dict2)
    else:
        current_Q = self.critic(obs, action)
        critic_loss, loss_dict = iq_loss(agent, current_Q, current_V, next_V, batch)

    # logger.log('train/critic_loss', critic_loss, step)
    log_dict = {}
    log_dict['train/critic_loss'] = critic_loss.item()
    logger.wandb_log(log_dict, step)

    # Optimize the critic
    self.critic_optimizer.zero_grad()
    critic_loss.backward()
    # step critic
    self.critic_optimizer.step()
    return loss_dict


def iq_update(self, policy_buffer, expert_buffer, logger, step):
    policy_batch = policy_buffer.get_samples(self.batch_size, self.device)
    expert_batch = expert_buffer.get_samples(self.batch_size, self.device)

    losses = self.iq_update_critic(policy_batch, expert_batch, logger, step)

    if self.actor and step % self.actor_update_frequency == 0:
        if not self.args.agent.vdice_actor:

            if self.args.offline:
                obs = expert_batch[0]
            else:
                # Use both policy and expert observations
                obs = torch.cat([policy_batch[0], expert_batch[0]], dim=0)

            if self.args.num_actor_updates:
                for i in range(self.args.num_actor_updates):
                    actor_alpha_losses = self.update_actor_and_alpha(obs, logger, step)

            losses.update(actor_alpha_losses)

    if step % self.critic_target_update_frequency == 0:
        if self.args.train.soft_update:
            soft_update(self.critic_net, self.critic_target_net,
                        self.critic_tau)
        else:
            hard_update(self.critic_net, self.critic_target_net)
    return losses


if __name__ == "__main__":
    main()
