import os
import time
from typing import Dict, List, Optional
from dataclasses import dataclass
import openai
from dotenv import load_dotenv

load_dotenv()

@dataclass
class LLMResponse:
    """Container for LLM response data."""
    content: str
    model: str
    prompt_tokens: int
    completion_tokens: int
    total_tokens: int
    response_time: float

class LLMClient:
    """Client for interacting with various LLM APIs."""
    
    def __init__(self, model: str = "gpt-41-mini-0414-global", api_key: Optional[str] = None):
        """
        Initialize the LLM client.
        
        Args:
            model: The model to use (e.g., "gpt-41-mini-0414-global")
            api_key: API key for the service. If None, will use environment variables.
        """
        self.model = model
        self.api_key = api_key or os.getenv("OPENAI_API_KEY")
        
        if not self.api_key:
            raise ValueError("API key not provided. Set OPENAI_API_KEY environment variable.")
        
        # Initialize OpenAI client with custom base URL if provided
        openai.api_key = self.api_key
        base_url = os.getenv("OPENAI_API_BASE", "")
        self.client = openai.OpenAI(api_key=self.api_key, base_url=base_url)
    
    def generate_response(
        self,
        messages: List[Dict[str, str]],
        max_tokens: Optional[int] = None,
        temperature: float = 0.7,
    ) -> LLMResponse:
        """Generate a response given a prepared messages list (no side-effect printing)."""
        if not isinstance(messages, list) or any(not isinstance(m, dict) for m in messages):
            raise ValueError("messages must be a list of dicts with 'role' and 'content'")
        start_time = time.time()
        try:
            response = self.client.chat.completions.create(
                model=self.model,
                messages=messages,
                max_tokens=max_tokens,
                temperature=temperature
            )
            response_time = time.time() - start_time
            content_text = response.choices[0].message.content
            return LLMResponse(
                content=content_text,
                model=self.model,
                prompt_tokens=response.usage.prompt_tokens,
                completion_tokens=response.usage.completion_tokens,
                total_tokens=response.usage.total_tokens,
                response_time=response_time
            )
        except Exception as e:
            raise RuntimeError(f"Error generating response: {str(e)}")


def create_llm_client(model: str = None, api_key: str = None) -> LLMClient:
    """
    Factory function to create an LLM client.
    
    Args:
        provider: The LLM provider ("openai" or "anthropic")
        model: The specific model to use
        api_key: API key for the service
        
    Returns:
        LLMClient instance
    """
    default_model = model or "gpt-41-mini-0414-global"
    return LLMClient(model=default_model, api_key=api_key)
