#!/bin/bash

# set -e

# make clean
# make

echo "Setting up performance stabilization..."
{
    # Set CPU governor to performance mode
    echo performance | tee /sys/devices/system/cpu/cpu*/cpufreq/scaling_governor 2>/dev/null && echo "CPU governor set to performance" || echo "CPU governor setting failed"
    
    # Set CPU affinity
    taskset -p -c 0 $$ 2>/dev/null && echo "CPU affinity set" || echo "CPU affinity setting failed"
    
    echo "Performance stabilization setup completed"
}

run() {
    for dist in "${distributions[@]}"; do
        for sort in "${sorts[@]}"; do
            for n in "${ns[@]}"; do
                # if [ "$sort" == "learned_AniKristo" ] && [ "$dist" == "data/sof_temp" ] && [ "$n" -gt 2000000 ]; then
                #     echo "Skip $sort $dist $n"
                #     continue
                # fi

                now_date_str=$(date '+%y%m%d%H%M%S')
                dist_clean=${dist//\//}
                dist_clean=${dist_clean//_/}
                result_date_dir="$RESULT_DIR/sort_time/${now_date_str}_${sort}_${dist_clean}_n${n}"

                if compgen -G "$RESULT_DIR/sort_time/*_${sort}_${dist_clean}_n${n}" > /dev/null; then
                    echo "Skip $sort $dist $n"
                    continue
                fi

                echo "Running $sort $dist $n"

                # Clear cache
                sudo sh -c "sync; echo 3 > /proc/sys/vm/drop_caches"

                mkdir -p "$result_date_dir"

                echo "{" > "$result_date_dir/args.txt"
                echo "  \"n\": $n," >> "$result_date_dir/args.txt"
                echo "  \"dist\": \"$dist\"," >> "$result_date_dir/args.txt"
                echo "  \"iterations\": $iterations," >> "$result_date_dir/args.txt"
                echo "  \"sort\": \"$sort\"," >> "$result_date_dir/args.txt"
                echo "  \"optflag\": \"$optflag\"" >> "$result_date_dir/args.txt"
                echo "}" >> "$result_date_dir/args.txt"

                command="./bin/measure_sort_time $sort $n $dist $iterations"
                echo "$command" > "$result_date_dir/command_$n.txt"
                timeout 3600 bash -c "$command" > "$result_date_dir/time_$n.txt"
                timeout_exit_code=$?

                if [ $timeout_exit_code -eq 124 ]; then
                    echo "[Timeout] Execution of $sort for $dist with size $n exceeded 1 hour limit."
                    echo "TIMEOUT" > "$result_date_dir/time_$n.txt"
                    # break
                elif [ $timeout_exit_code -ne 0 ]; then
                    echo "[Error] Execution of $sort for $dist with size $n failed."
                    rm -rf "$result_date_dir"
                fi
            done
            # set +e
            # python plot/plot_n_time.py
            # set -e
        done
    done
}

if [ -z "$RESULT_DIR" ]; then
    RESULT_DIR="result"
fi

iterations=10

distributions=(
    "uniform"
    "normal"
    "exponential"
    "lognormal"

    "uniform_shift"
    "normal_shift"
    "exponential_shift"
    "lognormal_shift"

    "data/wiki_ts_200M_uint64"
    "data/osm_cellids_800M_uint64"
    "data/books_800M_uint64"
    "data/fb_200M_uint64"

    "data/nyc_pickup"
    "data/nyc_dist"
    "data/nyc_tot"

    "data/stks_vol"
    "data/stks_open"
    "data/stks_date"
    "data/stks_low"

    "data/sof_hum"
    "data/sof_press"
    "data/sof_temp"

    "data/chic_start"
    "data/chic_tot"
)

ns=(
    10000
    20000
    50000
    100000
    200000
    500000
    1000000
    2000000
    5000000
    10000000
)

sorts=(
    "pcf"
    "std"
    "boost"
    "radix"
    "ips4o"
    "learned_AniKristo"
    "bls"
    "uls"
    "ls21"
    "learned_sort_using_learned_index_binary_search"
    "learned_sort_using_learned_index_btree"
    "learned_sort_using_learned_index_espc"
    "learned_sort_using_learned_index_pgm"
    "learned_sort_using_learned_index_rmi"
)

run
