#ifndef __PCF_LEARNED_SORT_H__
#define __PCF_LEARNED_SORT_H__

#include <cmath>
#include <vector>
#include <algorithm>
#include <functional>
#include <execution>
#include <iostream>

namespace pcf_learned_sort
{

// Threshold τ: arrays smaller than τ are handled by a standard sort
inline size_t default_tau = 100;

/*
    Default parameter policy for PCF bucketing.

    Parameters (paper terminology):
        - α (alpha): number of samples used to train the CDF model (PCF)
        - β (beta): number of intervals in the PCF
        - γ (gamma): number of buckets produced by model-based bucketing
        - δ (delta): switching threshold; buckets with size ≥ δ are sorted by a standard sort

    This policy instantiates the common setting α = β = γ = δ = n^{3/4},
    which is used in the theoretical analysis of PCF Learned Sort.
*/
inline std::tuple<size_t, size_t, size_t, size_t>
default_pcf_bucketing_alpha_beta_gamma_delta_func (const size_t n) {
    size_t n_75 = static_cast<size_t>(std::pow(n, 0.75));
    return std::make_tuple(n_75, n_75, n_75, n_75);
}

template <class Iterator>
void pcf_learned_sort(Iterator begin, Iterator end) {
    size_t n = std::distance(begin, end);

    // Base case (τ): use a standard sort with known worst/expected guarantees
    if (n < default_tau) {
        std::sort(std::execution::unseq, begin, end);
        return;
    }

    // ===================== Model-based Bucketing via PCF =====================
    using value_type = typename std::iterator_traits<Iterator>::value_type;

    // (α, β, γ, δ) as defined in the paper
    size_t alpha, beta, gamma, delta;
    std::tie(alpha, beta, gamma, delta) =
        default_pcf_bucketing_alpha_beta_gamma_delta_func(n);

    // Train the PCF CDF model: estimate range and prepare uniform-width intervals
    auto [min_it, max_it] = std::minmax_element(std::execution::unseq, begin, end);
    const double x_min = *min_it;
    const double x_max = *max_it;
    const double beta_inv_x_range = static_cast<double>(beta) / (x_max - x_min);
    const double gamma_inv_alpha = static_cast<double>(gamma) / alpha;

    // PCF training (using α samples): build a cumulative histogram over β intervals
    // b_ counts per interval; b is its inclusive prefix (an empirical CDF over intervals)
    std::vector<size_t> b_(beta + 1, 0);
    for (size_t i = 0; i < alpha; ++i) {
        size_t bin_id = static_cast<size_t>((*(begin + i) - x_min) * beta_inv_x_range);
        bin_id = std::min(bin_id, beta);
        ++b_[bin_id];
    }
    std::vector<size_t> b(b_.size());
    std::partial_sum(b_.begin(), b_.end(), b.begin());

    // Use the trained PCF to assign each element to one of γ+1 buckets
    std::vector<size_t> bucketSizes(gamma + 1, 0);
    std::vector<uint32_t> bucketIds(n);
    for (size_t i = 0; i < n; ++i) {
        size_t bin_id = static_cast<size_t>((*(begin + i) - x_min) * beta_inv_x_range);
        bin_id = std::min(bin_id, beta);
        uint32_t bucket_id = static_cast<uint32_t>(b[bin_id] * gamma_inv_alpha);
        bucketIds[i] = bucket_id;
        ++bucketSizes[bucket_id];
    }

    // Compute bucket offsets (prefix sums) and place elements into contiguous bucket ranges
    // This realizes the "partition then concatenate" structure used in the analysis
    std::vector<size_t> bucketOffsets((gamma + 1) + 2);
    bucketOffsets[0] = 0;
    bucketOffsets[1] = 0;
    std::partial_sum(bucketSizes.begin(), bucketSizes.end(), bucketOffsets.begin() + 2);

    std::vector<value_type> bins(n);
    for (size_t i = 0; i < n; ++i) {
        size_t pos = bucketOffsets[bucketIds[i] + 1]++;
        bins[pos] = *(begin + i);
    }
    // =================== End of Model-based Bucketing (PCF) ===================

    std::copy(std::execution::unseq, bins.begin(), bins.end(), begin);
    bins.clear();

    // Recursively process each bucket:
    //  - if size ≥ δ (bucketing "fails"), apply a standard sort
    //  - otherwise (bucketing "succeeds"), recurse with parameters re-instantiated for the subproblem
    for (size_t bucket_id = 0; bucket_id <= gamma; ++bucket_id) {
        auto bucket_begin = begin + bucketOffsets[bucket_id];
        auto bucket_end   = begin + bucketOffsets[bucket_id + 1];
        size_t bucket_size = std::distance(bucket_begin, bucket_end);

        if (bucket_size >= delta) {
            // Standard sort path (guaranteed worst/expected complexity)
            std::sort(std::execution::unseq, bucket_begin, bucket_end);
        } else {
            // Recursive model-based bucketing on a smaller interval
            pcf_learned_sort(bucket_begin, bucket_end);
        }
    }

    return;
}

} // namespace pcf_learned_sort

#endif
