from action import APIAction
import json
import uuid


class APIPrepare(APIAction):
    '''
        prepare natural language descriptions for the API tool
        including tool description, API description, and parameter description
    '''
    name: str = "api_prepare"
    fields: list = ["category_name", "tool_name", "api_name",  name]
    
    def __init__(self, profile, policy, handler):
        super(APIPrepare, self).__init__(profile, handler)

        self.policy = policy

    def build_tool_description_prompt(self, tool_info):
        prompt = self.prompt[self.name]["tool_description"].format(
            example=self.prompt[self.name]["tool_example"],
            example_desc=self.prompt[self.name]["tool_example_desc"],
            tool=tool_info,
        )
        return [{"role": "user", "content": prompt}]
    
    def build_api_description_prompt(self, api):
        prompt = self.prompt[self.name]["api_description"].format(
            example=self.prompt[self.name]["api_example"],
            example_desc=self.prompt[self.name]["api_example_desc"],
            function=api,
        )
        return [{"role": "user", "content": prompt}]

    def build_parameter_description_prompt(self, api):
        prompt = self.prompt[self.name]["param_description"].format(
            example=self.prompt[self.name]["param_example"],
            example_desc=self.prompt[self.name]["param_example_desc"],
            function=api,
        )
        return [{"role": "user", "content": prompt}]

    def build4apitext(self, api, api_db_param):  
        prompt = self.build_api_description_prompt(api=api)
        api_info = self.policy(prompt)
        api_info = api_info.replace("\\", "").replace("\\\\", "")
        
        param_prompt = self.build_parameter_description_prompt(api=api)
        param_info = self.policy(param_prompt)
        param_info = param_info.replace("\\", "").replace("\\\\", "")
        try:
            api_desc = {
                "api_info": api_info,
                "param_info": json.loads(param_info),
            }
            self.handler.save_action_feedback(None, self.name, self.name, api_desc, api_db_param)
        except:
            api_desc = {
                "api_info": api_info,
                "param_info": {},
            }
            self.handler.save_action_feedback(None, self.name, self.name, api_desc, api_db_param)
        return
        
    def load_api_text(self, category, tool_name, tool_info):
        
        build_args = []
        for api in tool_info['available_functions']:
            api_db_param = {
                "category_name": str(uuid.uuid3(uuid.NAMESPACE_DNS, category)),
                "tool_name": str(uuid.uuid3(uuid.NAMESPACE_DNS, tool_name)),
                "api_name": str(uuid.uuid3(uuid.NAMESPACE_DNS, api["name"]))
            }
            api_desc = self.handler.load_action_feedback(None, self.name, self.name, api_db_param)
            if api_desc is None:
                build_args.append((api, api_db_param))
            else:
                api['description'] = api_desc['api_info']
                for k, v in api_desc['param_info'].items():
                    if k in api["parameters"]["properties"]:
                        if 'description' not in api["parameters"]["properties"][k]:
                            api["parameters"]["properties"][k]['description'] = v
                                
        return build_args
    
    def run(self, state, message, json_format):
        '''
            Load the API list from ToolBench candidate apis
            The format is as following:
            {
                "category": cate_name,
                "description": "tool text description",
                "available_functions": [
                    {
                        "ID": tool_idx,
                        "name": "get_current_weather",
                        "description": "api text description",
                        "parameters": {
                            "properties": {
                                "location": {
                                    "description": "parameter text description",
                                    "type": "string"
                                },
                            },
                            "required": [
                                "location"
                            ],
                            "type": "object"
                        }
                    }
                ]
            }
        '''
        available_api_dict = self.load_available_api(message['api_list'])
        
        if json_format:
            return available_api_dict
        
        for tool_name, tool_info in available_api_dict.items():
            
            
            # 1. build tool description
            tool_db_param = {
                "category_name": str(uuid.uuid3(uuid.NAMESPACE_DNS, tool_info["category"])),
                "tool_name": str(uuid.uuid3(uuid.NAMESPACE_DNS, tool_name)),
                "api_name": str(uuid.uuid3(uuid.NAMESPACE_DNS, "pesudo_api_name"))
            }
            tool_desc = self.handler.load_action_feedback(None, self.name, self.name, tool_db_param)
            
            if tool_desc is None:
                prompt = self.build_tool_description_prompt(tool_info)
                tool_description = self.policy(prompt)
                tool_desc = tool_description.replace("\\", "")
                self.handler.save_action_feedback(None, self.name, self.name, tool_desc, tool_db_param)
            available_api_dict[tool_name]["description"] = tool_desc

            # 2. build api description and parameter description
            build_args = [1]
            max_loop = 3
            while len(build_args) and max_loop:
                build_args = self.load_api_text(tool_info["category"], tool_name, tool_info)
                
                # process multiple requests in parallel
                self.action_runner.run([self.build4apitext] * len(build_args), build_args)
                max_loop -= 1

        return available_api_dict