import yaml, os, json
from utils import api_util, log
import requests
import time
from openai import OpenAI
import random


class Rapid:

    def __init__(self, config_file):
        
        self.logger = log.get_loguru()

        CONFIG = yaml.load(open(config_file, 'r'), Loader=yaml.FullLoader)
        self.tool_dir = CONFIG['tool_dir']
        self.category_set_file = CONFIG['category_set_file']

        self.client = OpenAI(
            base_url = "",
            api_key = ""
        )
        self.cache = {}

        self.black_list = set()
        for line in open("all_zero.jsonl").readlines():
            self.black_list.add(json.loads(line)['tool_name'])
            # import pdb; pdb.set_trace()

    def get_all_category(self):
        
        if "category" not in self.cache:
            category = {}
            for dir_name in os.listdir(self.tool_dir):
                dir_path = os.path.join(self.tool_dir, dir_name)
                category[dir_name] = dir_path

            self.logger.info("rapid category number {}".format(len(category)))
            self.cache['category'] = category
        return self.cache['category']

    def get_tool_by_category(self, category_path):
        
        if "tool_dict" not in self.cache:
            self.cache['tool_dict'] = {}
        
        if category_path not in self.cache['tool_dict']:
            tool_dict = {}
            for file in os.listdir(category_path):
                if file.endswith(".json"):
                    file_path = os.path.join(category_path, file)
                    tool_dict[file.replace(".json", "")] = file_path
            self.cache['tool_dict'][category_path] = tool_dict
        
        return self.cache['tool_dict'][category_path]
    
    def get_api_info(self, category_name, tool_name, api_name):
        
        category_dict = self.get_all_category()
        tool_dict = self.get_tool_by_category(category_dict[category_name])
        api_dict = self.get_api(category_name, tool_dict[tool_name])
        return api_dict[api_name]

    def get_candidate_api(self, category_name, tool_name, api_name, max_number=3):
        
        candidates = []
        category_dict = self.get_all_category()
        tool_dict = self.get_tool_by_category(category_dict[category_name])
        api_dict = self.get_api(category_name, tool_dict[tool_name])

        for candi_api_name, api_info in api_dict.items():
            if api_name != candi_api_name and len(candidates) < max_number:
                candidates.append(api_info)
            if len(candidates) >= max_number:
                return candidates
        
        tool_name_list = list(tool_dict.keys())
        random.shuffle(tool_name_list)
        for candi_tool_name in tool_name_list:
            for candi_api_name, api_info in self.get_api(category_name, tool_dict[candi_tool_name]).items():
                if api_name != candi_api_name and len(candidates) < max_number:
                    candidates.append(api_info)
                if len(candidates) >= max_number:
                    return candidates
        
        return candidates

    def get_api(self, category_name, tool_path):
        
        tool_data = json.load(open(tool_path))
        api_dict = {}

        for api_item in tool_data["api_list"]:
            
            api_dict[api_item["name"]] = {
                "category_name": category_name,
                "tool_name": tool_data["tool_name"],
                "tool_description": tool_data["tool_description"],
                "api_name": api_item["name"],
                "api_description": api_item["description"],
                "required_parameters": api_item["required_parameters"],
                "optional_parameters": api_item["optional_parameters"],
                "method": api_item["method"]
            }
        
        return api_dict
        
    def request_api(self, category_name, tool_name, api_name, api_input, return_all=False):

        start_time = time.perf_counter()
        category_name_slim = api_util.standardize_category(category_name)
        tool_name_slim = api_util.standardize(tool_name)
        api_name_slim = api_util.change_name(api_name)
        
        if tool_name_slim in self.black_list or tool_name in self.black_list:
            return False, "Unsubscribed error"

        payload = {
            "category": category_name,
            "tool_name": tool_name,
            "api_name": api_name,
            "tool_input": api_input,
            "toolbench_key": "",
        }
            
        headers = {
            'accept': 'application/json',
            'Content-Type': 'application/json',
        }

        try:
            response = requests.post("", json=payload, headers=headers, timeout=60)
        except Exception as e :
            self.logger.info("Rapid API {} {} {}.\nException: {}...".format(category_name, tool_name, api_name, e))
            return False, "time_out"
        
        end_time = time.perf_counter()
        elapsed_time = end_time - start_time

        self.logger.info("Rapid API {} {} {}. Call Sucess...".format(category_name, tool_name, api_name))
        
        if response.status_code == 200:
            response_text = response.json()
            if response_text['error']:
                self.logger.info("Rapid API {} {} {}. Call Error {}...".format(category_name, tool_name, api_name, response_text['error']))
                return False, response_text
            else:
                if self.response_check(response_text['response']):
                    if return_all:
                        return True, response_text['response'][:4096]
                    else:
                        return True, response_text['response'][:64]
                else:
                    return False, response_text['response'][:64]
                
        else:
            return False, "Response Status: {}".format(response.status_code)

    def response_check(self, response_text):

        if "does not exist" in response_text \
            or "Unauthorized" in response_text \
            or "Token Error" in response_text \
            or "No authentication" in response_text \
            or "Invalid access token" in response_text \
            or response_text == "" \
            or response_text == "None" \
            or response_text == "[]" \
            or "!DOCTYPE" in response_text \
            or "does not exist" in response_text \
            or "Service Unavailable" in response_text \
            or "not found for parameter" in response_text \
            or "API doesn't exists" in response_text \
            or "Internal Server Error" in response_text \
            or "404 Not Found" in response_text \
            or "Base URL is missing or is not configured prope" in response_text \
            or "Can't find" in response_text \
            or "<?xml version=\"1.0\" encoding=" in response_text \
            or "Invalid API key. Go to https://docs.rapidapi" in response_text \
            or "500 - Server Error" in response_text \
            or "404 not found" in response_text \
            or "'ErrorCode': 370, 'ErrorMessage': 'OrderId required'" in response_text \
            or "Error conectando a la base de datos" in response_text \
            or "This endpoint is disabled for your subscription" in response_text \
            or "Invalid API key provided." in response_text \
            or "<svg xmlns=\"http:" in response_text \
            or "'error': 'Not found'" in response_text \
            or "{'error': {'message': 'Authorization header" in response_text \
            or "'statusCode': 400, 'message': 'Experiment" in response_text \
            or "'status': 'error', 'code': 404, 'message': '" in response_text \
            or "'error': 'Not Found'" in response_text \
            or "'error': 'API v2 deprecated.'" in response_text \
            or "'status': 'error', 'message': ''" in response_text \
            or "'messages': 'Base URL is missing or is not configured" in response_text \
            or "'code': 'INVALID_TOKEN', 'message': 'Missing or invalid" in response_text \
            or "File not found." in response_text \
            or "JFIF" in response_text \
            or "'message': 'This endpoint is disabled for your subscription" in response_text \
            or "'notice': 'This endpoint is available on the Ultra plan" in response_text \
            or "<?xml version=\"1.0\"?>" in response_text \
            or "The requested resource could not be found" in response_text \
            or "'status': 'error', 'code': 404, 'message': 'Application" in response_text \
            or "'error': 'Page not found'" in response_text \
            or "404 page not found" in response_text \
            or "'detail': 'Object not found'" in response_text \
            or "'status': 'invalid_user'" in response_text \
            or "403 - Forbidden | Access to this page is forbidden." in response_text \
            or "'status': 'error', 'message': 'Token not found.'" in response_text \
            or "Expired API key" in response_text \
            or "Not Found" in response_text \
            or "<html>" in response_text:
            
            return False
        else:
            return True

    def guess_parameter(self, api_dict, default=True):

        if default:

            default_parameter = {}
            for param in api_dict["required_parameters"]:
                if "default" not in param:
                    break
                    
                default_parameter[param['name']] = param['default']
            
            return default_parameter
        
        else:
            raise Exception("Not support this method...")
    
    def request_gpt(self, query):

        messages = [{"role": "user", "content": query}]

        chat_completion = self.client.chat.completions.create(
            model="gpt-4o-mini",
            messages=messages
        )
        return chat_completion.choices[0].message.content
