import yaml, os, json
from utils import api_util, log
import requests
import time
from openai import OpenAI
import random
from datetime import datetime


class RapidEnv:

    def __init__(self, profile):
        
        self.logger = log.get_loguru()

        parameter = profile.load_env()['rapid']
        self.timeout = parameter['timeout']
        self.url = parameter['url']
        self.toolbench_key = parameter['toolbench_key']

        self.debug = False
    
    @property
    def name(self):
        return "rapid"

    def __call__(self, category_name, tool_name, api_name, func_input, fmt=False, check=False):
        
        if self.debug:
            start_time = time.perf_counter()
        
        if fmt:
            category_name = api_util.standardize_category(category_name)
            tool_name = api_util.standardize(tool_name)
            api_name = api_util.change_name(api_util.standardize(api_name))
            
        try:
            call_response = self._call(category_name, tool_name, api_name, func_input)
            # import pdb;pdb.set_trace()
        except Exception as e:
            self.logger.info("Exception {}".format(e))
            return False, "Exception {}".format(e)

        if self.debug:
            end_time = time.perf_counter()
            elapsed_time = end_time - start_time
        
        req_status = False
        if call_response.status_code == 200:
            
            response = call_response.json()
            if check:
                if response['error']:
                    self.logger.info("Rapid API {} {} {}. Call Error {}...".format(category_name, tool_name, api_name, response['error']))
                    req_status = False
                else:
                    if self._check(response['response']):
                        req_status = True
                    else:
                        req_status = False
            else:
                req_status = True
            
            return req_status, response
        else:
            return False, "Run Rapid Env Response Status={}".format(call_response.status_code)
    
    def _call(self, category_name, tool_name, api_name, func_input):
        
        payload = {
            "category": category_name,
            "tool_name": tool_name, 
            "api_name": api_name,
            "tool_input": func_input,
            "strip":"",
            # "platform": "rapidapi",
            "toolbench_key": self.toolbench_key,
        }
        
        headers = {
            'accept': 'application/json',
            'Content-Type': 'application/json',
        }
        
        return requests.post(self.url, json=payload, headers=headers, timeout=self.timeout)

    def _check(self, response_text):

        if "does not exist" in response_text \
            or "Unauthorized" in response_text \
            or "Token Error" in response_text \
            or "No authentication" in response_text \
            or "Invalid access token" in response_text \
            or response_text == "" \
            or response_text == "None" \
            or response_text == "[]" \
            or "!DOCTYPE" in response_text \
            or "does not exist" in response_text \
            or "Service Unavailable" in response_text \
            or "not found for parameter" in response_text \
            or "API doesn't exists" in response_text \
            or "Internal Server Error" in response_text \
            or "404 Not Found" in response_text \
            or "Base URL is missing or is not configured prope" in response_text \
            or "Can't find" in response_text \
            or "<?xml version=\"1.0\" encoding=" in response_text \
            or "Invalid API key. Go to https://docs.rapidapi" in response_text \
            or "500 - Server Error" in response_text \
            or "404 not found" in response_text \
            or "'ErrorCode': 370, 'ErrorMessage': 'OrderId required'" in response_text \
            or "Error conectando a la base de datos" in response_text \
            or "This endpoint is disabled for your subscription" in response_text \
            or "Invalid API key provided." in response_text \
            or "<svg xmlns=\"http:" in response_text \
            or "'error': 'Not found'" in response_text \
            or "{'error': {'message': 'Authorization header" in response_text \
            or "'statusCode': 400, 'message': 'Experiment" in response_text \
            or "'status': 'error', 'code': 404, 'message': '" in response_text \
            or "'error': 'Not Found'" in response_text \
            or "'error': 'API v2 deprecated.'" in response_text \
            or "'status': 'error', 'message': ''" in response_text \
            or "'messages': 'Base URL is missing or is not configured" in response_text \
            or "'code': 'INVALID_TOKEN', 'message': 'Missing or invalid" in response_text \
            or "File not found." in response_text \
            or "JFIF" in response_text \
            or "'message': 'This endpoint is disabled for your subscription" in response_text \
            or "'notice': 'This endpoint is available on the Ultra plan" in response_text \
            or "<?xml version=\"1.0\"?>" in response_text \
            or "The requested resource could not be found" in response_text \
            or "'status': 'error', 'code': 404, 'message': 'Application" in response_text \
            or "'error': 'Page not found'" in response_text \
            or "404 page not found" in response_text \
            or "'detail': 'Object not found'" in response_text \
            or "'status': 'invalid_user'" in response_text \
            or "403 - Forbidden | Access to this page is forbidden." in response_text \
            or "'status': 'error', 'message': 'Token not found.'" in response_text \
            or "Expired API key" in response_text \
            or "Not Found" in response_text \
            or "<html>" in response_text:
            
            return False
        else:
            return True

    def check_param(self, category_name, tool_name, api_name, func_input, fmt=False):
        
        if fmt:
            category_name = api_util.standardize_category(category_name)
            tool_name = api_util.standardize(tool_name)
            api_name = api_util.change_name(api_util.standardize(api_name))
        
        try:
            call_response = self._call_check_param(category_name, tool_name, api_name, func_input)
        except Exception as e:
            self.logger.info("Exception {}".format(e))
            return False, {"error": "Call: {}".format(e), "response": ""}
        
        if call_response.status_code == 200:
            response = call_response.json()
            if response['error']:
                return False, response
            else:
                return True, response
        else:
            return False, {"error": "Status: {}".format(call_response.status_code), "response": ""}

    def _call_check_param(self, category_name, tool_name, api_name, func_input):
        
        payload = {
            "category": category_name,
            "tool_name": tool_name, 
            "api_name": api_name,
            "tool_input": func_input,
            "check_parameters": True,
            "platform": "rapidapi",
            "toolbench_key": self.toolbench_key,
        }
        
        headers = {
            'accept': 'application/json',
            'Content-Type': 'application/json',
        }
        
        return requests.post(self.url, json=payload, headers=headers, timeout=self.timeout)
    
    def _is_valid_date(self, date_str):
        datetime.strptime(date_str, '%Y-%m-%d')

    def _is_valid_time(self, time_str):
        datetime.strptime(time_str, '%H:%M')

    def _type_format(self, p_value, p_type):
        
        if p_type is None:
            return None
        if p_value == "":
            return None
        
        if p_type == "NUMBER":
            try:
                p_value = int(p_value)
            except:
                p_value = float(p_value)
        elif p_type == "STRING" or p_type == 'string':
            p_value = str(p_value)
        elif p_type == "BOOLEAN":
            p_value = bool(p_value)
        elif p_type == "DATE (YYYY-MM-DD)":
            if self._is_valid_date(p_value):
                p_value = str(p_value)
        elif p_type == "TIME (24-hour HH:MM)":
            if self._is_valid_time(p_value):
                p_value = str(p_value)
        elif p_type == "OBJECT":
            p_value = str(p_value)
        elif p_type == "ENUM":
            p_value = str(p_value)
        elif p_type == "ARRAY":
            p_value = str(p_value)
        else:
            raise Exception("Not Support Type Format {}".format(p_type))

        return p_value

    def parameter_format(self, api_info, parameter):

        if len(api_info['required_parameters']):
            for p in api_info['required_parameters']:
                if p['name'] in parameter:
                   parameter[p['name']] = self._type_format(parameter[p['name']], p['type'])
                elif p['name'].lower() in parameter:
                   parameter[p['name'].lower()] = self._type_format(parameter[p['name'].lower()], p['type'])
        
        if len(api_info['optional_parameters']):
            for p in api_info['optional_parameters']:
                if p['name'] in parameter:
                   parameter[p['name']] = self._type_format(parameter[p['name']], p['type'])
                elif p['name'].lower() in parameter:
                   parameter[p['name'].lower()] = self._type_format(parameter[p['name'].lower()], p['type'])
        
        return parameter