from parallel.runner import InstanceQueue
from profile.registry import ProfileRegistry
from pipeline.rapid_sampler import RapidSampler
from pipeline.leetcode_sampler import LeetcodeSampler
from utils import log
from enum import Enum
import argparse


def run(args):

    parallel_number = args.parallel_number
    logger = log.get_loguru()
    logger.info(
        "Run API Sampler with <{}>".format(
            "Parallel Execution" if parallel_number > 1 else "Step-by-Step Execution..."
        )
    )
    
    profile = ProfileRegistry.create_instance("sampler", args)
    
    if args.env == Env.Rapid:
        sampler = RapidSampler(profile)
    elif args.env == Env.Leetcode:
        sampler = LeetcodeSampler(profile)
    
    data_generator = sampler.load_single_generator()

    if args.action == Action.Single: 
        exector = sampler.run_single
    elif args.action == Action.Parallel:
        exector = sampler.run_parallel
    elif args.action == Action.Multiple:
        exector = sampler.run_multiple
    
    if parallel_number > 0:
        instance_queue = InstanceQueue(parallel_number, asyncio=False)
        instance_queue.run_param(exector, data_generator)
    else:
        for idx, d in enumerate(data_generator):
            exector(d)

    if args.action == Action.Multiple:
        sampler.run_multiple_sampling()

    logger.info("Run API Sampler End")


class Env(Enum):

    Rapid = 'rapid'
    Leetcode = 'leetcode'

class Action(Enum):

    Single = "single"
    Parallel = "parallel"
    Multiple = "multiple"
    MultipleParallel = "multiple_parallel"

if __name__ == "__main__":
    
    parser = argparse.ArgumentParser(description="auto collector")
    parser.add_argument(
        "-c",
        "--config-file",
        type=str,
        default="profiles/rapid_profile/collector/config.json",
        help="The file of configuration.",
    )
    parser.add_argument(
        "-p",
        "--parallel-number",
        type=int,
        default=1,
        help="The number of concurrent API calls.",
    )
    parser.add_argument(
        "-e",
        '--env',
        type=Env,
        choices=list(Env),
        required=True,
        help="choose a env from list"
    )
    parser.add_argument(
        "-a",
        '--action',
        type=Action,
        choices=list(Action),
        required=True,
        help="choose a action from list"
    )
    args = parser.parse_args()
    run(args)
