"""General helper functions.

Currently provides deterministic seeding utilities.
"""

from __future__ import annotations

import random
from typing import Optional

import numpy as np
import torch


def seed_everything(seed: int, *, deterministic: bool = True) -> None:
    """Seed Python, NumPy, and PyTorch for reproducibility.

    Args:
        seed: Base integer seed.
        deterministic: If True, enable deterministic cuDNN settings.

    Time/Space: O(1) time, O(1) space.
    """
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)
    np.random.seed(seed)
    random.seed(seed)
    if deterministic:
        torch.backends.cudnn.benchmark = False
        torch.backends.cudnn.deterministic = True


