"""Global timing utilities for evaluation and attention profiling.

This module provides the GlobalTimer class to measure wall-clock times for
prefill and decode phases, as well as fine-grained sections such as KV append,
index search, attention computation, and MLP projection.

Usage:
    timer = GlobalTimer()
    timer.reset()
    # ... during model run, update fields or use hooks to accumulate times ...

All comments and docstrings are in English per project rules.
"""

from __future__ import annotations

from dataclasses import dataclass


@dataclass
class GlobalTimer:
    """Global timer with accumulators.

    Note: All fields are public simple accumulators to minimize overhead.
    """

    prefill_start: float | None = None
    prefill_end: float | None = None
    decode_total: float = 0.0
    last_decode_start: float | None = None

    append_time: float = 0.0
    search_time: float = 0.0
    skip_search_time: float = 0.0
    attn_time: float = 0.0
    mlp_time: float = 0.0
    attnetion_layer_total: float = 0.0

    is_sparse: bool = True
    cur_max_length: int = 0

    def reset(self) -> None:
        """Reset all accumulators to initial state.

        Time/Space: O(1) time, O(1) space.
        """
        self.prefill_start = None
        self.prefill_end = None
        self.decode_total = 0.0
        self.last_decode_start = None
        self.append_time = 0.0
        self.search_time = 0.0
        self.skip_search_time = 0.0
        self.attn_time = 0.0
        self.mlp_time = 0.0
        self.attnetion_layer_total = 0.0
        self.cur_max_length = 0


