from z3 import *
import numpy as np

def BinairoSolver(input_board, **kwargs):
    N = len(input_board)
    half_n = N // 2
    s = Solver()
    # Create Z3 variables
    variables = [[Int(f'cell_{i}_{j}') for j in range(N)] for i in range(N)]
    # Constraints for known values
    for i in range(N):
        for j in range(N):
            if input_board[i][j] == 0 or input_board[i][j] == '0':
                s.add(variables[i][j] == 0)
            elif input_board[i][j] == 1 or input_board[i][j] == '1':
                s.add(variables[i][j] == 1)
            else:
                s.add(variables[i][j] >= 0)
                s.add(variables[i][j] <= 1)
    
    # Rule 1: Equal number of 1's and 0's in each row and column
    for i in range(N):
        s.add(Sum(variables[i]) == half_n)  # Rows
        s.add(Sum([variables[j][i] for j in range(N)]) == half_n)  # Columns 

    # Rule 2: No more than two identical numbers next to each other
    for i in range(N):
        for j in range(N - 2):
            s.add(Or(variables[i][j] + variables[i][j+1] + variables[i][j+2] == 1,
                     variables[i][j] + variables[i][j+1] + variables[i][j+2] == 2))
            s.add(Or(variables[j][i] + variables[j+1][i] + variables[j+2][i] == 1,
                     variables[j][i] + variables[j+1][i] + variables[j+2][i] == 2))
    
    # Rule 3: No two rows or columns can be identical
    for i in range(N):
        for j in range(i+1, N):
            # For rows
            s.add(Or([variables[i][k] != variables[j][k] for k in range(N)]))
            # For columns
            s.add(Or([variables[k][i] != variables[k][j] for k in range(N)]))

    # Check if solution exists
    if s.check() == sat:
        m = s.model()
        # Extract solution
        solution = []
        for i in range(N):
            solution.append([m.evaluate(variables[i][j]).as_long() for j in range(N)])
        return [solution]
    else:
        raise None
    
def MySolver():
    return BinairoSolver

if __name__ == "__main__":  
    input_board = [['?', '?', '?', '?', '?', '?', '?','?'],
                 [1, '?', '?', '?', '?', '?', '?', '?'],
                 ['?', '?', 0, '?', '?', '?', '?', '?'],
                 ['?' ,'?', 1, '?', '?', '?', '?', '?'],
                 ['?', 1, '?', '?', '?', '?', '?', 0],
                 [0, '?', 0, '?', '?', '?', '?', '?'],
                 [1, 0, 1, '?', '?', 0, '?', 0],
                 ['?', '?', '?', '?', '?', '?', '?', '?']]
    print(input_board)
    
    output_board = BinairoSolver(input_board)[0]
    print(output_board)
    

