from z3 import *

def CarSequencingSolver(input_sample, **kwargs):
    num_cars = input_sample['num_cars']
    num_options = input_sample['num_options']
    num_classes = input_sample['num_classes']
    cars_per_option = input_sample['cars_per_option']
    block_sizes = input_sample['block_sizes']
    cars_per_class = input_sample['cars_per_class']
    class_data = input_sample['class_data']

    # Variables
    class_of_slot = [Int(f"S_{i}") for i in range(num_cars)] ## class of car in slot i
    ## o[i][j] is True if slot i requires option j
    option_required_for_slot  =  [[Bool(f"O_{i}_{j}") for j in range(num_options)] for i in range(num_cars)]

    # Constraints
    constraints = []

    for i in range(num_cars): ## each slot should have one class
        constraints.append(class_of_slot[i] >= 1)
        constraints.append(class_of_slot[i] <= num_classes)

    for car_class in range(num_classes): ## each class should have required number of cars
        constraints.append(Sum([If(class_of_slot[slot] == car_class+1, 1, 0) for slot in range(num_cars)]) == cars_per_class[car_class])
        
    for slot in range(num_cars): ## each slot
        for car_class in range(num_classes):
            for option in range(num_options):
                constraints.append(Implies(And(class_of_slot[slot] == car_class+1, class_data[car_class][option]==1),option_required_for_slot[slot][option]))
                

    for option in range(num_options):
        for start_idx in range(0, num_cars-block_sizes[option]+1):
            constraints.append(Sum([option_required_for_slot[start_idx+b][option] for b in range(block_sizes[option])]) <= cars_per_option[option])

    # Solver setup
    s = Solver()
    s.add(constraints)

    if s.check() == sat:
        m = s.model()
        result = [[m.evaluate(class_of_slot[i]).as_long() for i in range(num_cars)]]
        return result
    else:
        print("No solution exists")


def MySolver():
    return CarSequencingSolver

if __name__ == "__main__":
    # Sample Input
    input_sample = {
        'num_cars': 10,
        'num_options': 5,
        'num_classes' : 6,
        'cars_per_option': [1, 2, 1, 2, 1],
        'block_sizes': [2, 3, 3, 5, 5],
        'cars_per_class': [1, 1, 2, 2, 2, 2],
        'class_data': [
            [1, 0, 1, 1, 0], ### options required by class i
            [0, 0, 0, 1, 0],
            [0, 1, 0, 0, 1],
            [0, 1, 0, 1, 0],
            [1, 0, 1, 0, 0],
            [1, 1, 0, 0, 0]
        ]
    }
    print(CarSequencingSolver(input_sample))