import numpy as np

EMPTY = 0
FILLED = 1
BALLOON = 2
IRON_BALL = 3

def DosunFuwariVerifier(input_sample, output_board, **kwargs):
    """
        0 => empty cell
        1 => filled cell
        2 => ballon (light)
        3 => iron ball (heavy)
    """
    input_board, subgrids = input_sample['input_board'], input_sample['subgrids']
    input_board = np.array(input_board)
    output_board = np.array(output_board)
    if input_board.shape != output_board.shape:
        return {
            'result': False,
            'reason': 'Input Board and Output Board Must be of the Same Shape'
        }
        
    assert input_board.shape[0] == input_board.shape[1]
    N = input_board.shape[0]
    
    for i in range(N):
        for j in range(N):
            if input_board[i, j] != EMPTY and output_board[i, j] != input_board[i, j]:
                return {
                    'result': False,
                    'reason': f'Original Elements on Non-Empty cells in the Input Board cannot be changed in the Output Board. Eg. Cell ({i}, {j}) (0-indexed) was changed in the output'
                }
            if output_board[i, j] == FILLED and input_board[i, j] != FILLED:
                return {
                    'result': False,
                    'reason': f"Output Board cannot have filled cells not in the input board. Cell ({i}, {j}) (0-indexed) was filled in the output but was not filled in the input"
                }
    valid_elements = np.all(np.isin(output_board.flatten(), np.arange(0,4)))
    if not valid_elements:
        return {
            'result': False,
            'reason': "Output Board can only have elements from 0-3"
        }

    
    for subgrid_idx, subgrid in enumerate(subgrids):
        balloon_count, iron_ball_count = 0, 0
        for cell in subgrid:
            x, y = cell//N, cell%N
            
            if output_board[x, y] == BALLOON:
                ## balloon
                balloon_count += 1
                if x == 0:
                    continue
                top_cell_x, top_cell_y = x-1, y
                if output_board[top_cell_x, top_cell_y] == EMPTY or output_board[top_cell_x, top_cell_y] == IRON_BALL:
                    return {
                        'result': False,
                        'reason': f"Balloons are light and float, so they must be placed in one of the cells at the top, or in a cell right under a black cell (filled-in cell) or under other balloons. Constraint Violated for Balloon at Cell ({x}, {y}) (0-indexed) in the output"
                    }
            
            if output_board[x, y] == IRON_BALL:
                ## iron ball
                iron_ball_count += 1
                if x == N-1:
                    continue ### bottom
                bottom_cell_x, bottom_cell_y = x+1, y
                if output_board[bottom_cell_x, bottom_cell_y] == EMPTY or output_board[bottom_cell_x, bottom_cell_y] == BALLOON:
                    return {
                        'result': False,
                        'reason': f"Iron balls are heavy and sink, so they must be placed in one of the cells at the bottom, or in a cell right over a black cell or over other iron balls. Constraint Violated for Iron Ball at Cell ({x}, {y}) (0-indexed) in the output"
                    }
                
        if balloon_count != 1 or iron_ball_count != 1:
            return {
                'result': False,
                'reason': f"Each subgrid must have one ballon and one iron ball exactly. Constraint Violated for subgrid {subgrid}"
            }
    
    return {
        'result': True,
        'reason': None
    }

def MyVerifier():
    return DosunFuwariVerifier

if __name__ == '__main__':
    print(DosunFuwariVerifier(input_sample={
                                    'input_board': [
                                                        [0,0,0,0],
                                                        [0,1,0,0],
                                                        [0,0,0,0],
                                                        [0,0,0,1]
                                                    ],
                                    'subgrids': [
                                                    [0, 1],
                                                    [2, 3, 6, 10],
                                                    [4, 8, 12],
                                                    [5, 9, 13, 14, 15],
                                                    [7, 11]
                                                ]
    }, output_board= [
                        [2,3,0,2],
                        [2,1,0,2],
                        [0,2,3,3],
                        [3,0,3,1]
                    ]))

    print(DosunFuwariVerifier(input_sample={
                                    'input_board': [
                                                        [0,0,0],
                                                        [0,0,1],
                                                        [0,0,0]
                                                    ],
                                    'subgrids': [
                                                    [0, 1, 2, 5],
                                                    [3, 4],
                                                    [6, 7, 8],
                                                ]}
                              , output_board= [
                                                [3,2,0],
                                                [3,2,1],
                                                [3,0,2]
                                              ]
                              ))
            
            
        
                