from z3 import *

def FactoryWorkersSolver(input_sample, **kwargs):
    factories_workers = input_sample['factories_workers']
    workers_jobs = input_sample['workers_jobs']
    
    num_factories = len(factories_workers)
    num_jobs = len(workers_jobs)
    num_workers = max(max(workers) for workers in factories_workers)
    solver = Solver()
    
    # Creating variables: done_worker_job[j][w] is True if worker w is doing job j
    done_worker_job = [[Bool(f"done_worker_{w}_job_{j}") for w in range(num_workers)] for j in range(num_jobs)]
    
    # Constraint 1: Each job must be done by at least one worker capable of doing it
    for j in range(num_jobs):
        solver.append(Or([done_worker_job[j][worker-1] for worker in workers_jobs[j]]))
    
    # Constraint 2: Each worker can do at most one job
    for w in range(num_workers):
        solver.add(Sum([If(done_worker_job[j][w], 1, 0) for j in range(num_jobs)]) <= 1)
    
    # Constraint 3: Each factory dispatches at most one worker
    for i in range(num_factories):
       workers_dispatched = [Sum([done_worker_job[j][worker-1] for j in range(num_jobs)]) for worker in factories_workers[i]]
       solver.add(Sum(workers_dispatched) <= 1)
    
    # Check if the model is satisfiable (i.e., there is a way to assign workers to jobs according to the rules)
    if solver.check() == sat:
        model = solver.model()
        # print([[model.evaluate(done_worker_job[j][w]) for w in range(num_workers)] for j in range(num_jobs)])
        return ["YES"]
    else:
        return ["NO"]

def MySolver():
    return FactoryWorkersSolver

if __name__ == '__main__':
    input_sample = {
        'factories_workers': [[1, 2], [3, 4], [5, 6]], ## each list is a factory and ID of workers working in that factory
        'workers_jobs': [[1, 3, 5], [2, 4, 6], [1, 6]] ## each list is a job and worker IDs capable of doing it
    }
    result = FactoryWorkersSolver(input_sample)
    print("Can the jobs be completed?", result)
