from z3 import *

def HamiltonianPathSolver(input_sample, **kwargs):
    N = input_sample['number_of_nodes']
    edges = set(input_sample['edges'])
    
    solver = Solver()

    # these variables represent the path
    # the ith variable represents the node number of the ith node on the path
    variables = [Int(f'x_{i}') for i in range(N)]
    
    ### initial constraints
    for i in range(N):
        solver.add(variables[i] >= 0, variables[i] <= N-1)
    solver.add(Distinct(*variables))
    
    for idx in range(N-1):
        ### for every adjacent neighbors in the path ensure that they are edge
        ### contrapositive if two nodes are not connected then they cannot be adjacent
        for x in range(N):
            for y in range(x+1):
                if (x, y) not in edges and (y, x) not in edges:
                    solver.add(Not(And(variables[idx] == x, variables[idx+1] == y)))
                    solver.add(Not(And(variables[idx] == y, variables[idx+1] == x)))    
    
    if solver.check() == sat:
        return ["YES"]
    else:
        return ["NO"]

def MySolver():
    return HamiltonianPathSolver

if __name__ == "__main__":
    input_sample = {
        'number_of_nodes': 5,
        'edges': [(0, 1), (1, 2), (2, 3), (3, 4)]
    }
    print(HamiltonianPathSolver(input_sample)) ## YES

    input_sample = {
        'number_of_nodes': 4,
        'edges': [(0, 1), (0, 2), (1, 2), (2, 3), (1, 3)]
    }
    print(HamiltonianPathSolver(input_sample)) ## YES

    input_sample = {
        'number_of_nodes': 4,
        'edges': [(0, 1), (0, 2), (0, 3)]
    }
    print(HamiltonianPathSolver(input_sample)) ### NO

    input_sample = {
        'number_of_nodes': 4,
        'edges': [(0, 1), (1, 2), (2, 3), (1, 3)]
    }
    print(HamiltonianPathSolver(input_sample)) ### YES

    input_sample = {
        'number_of_nodes': 6,
        'edges': [(0, 1), (1, 2), (2, 3), (2, 4), (4, 5)]
    }
    print(HamiltonianPathSolver(input_sample)) ### NO