import numpy as np


def InshiNoHeyaVerifier(input_sample, output_board, **kwargs):
    """
    Args:
        input_sample: Dict{
            input_board (list[list[int]]): partially filled input board, empty cells denoted by zero
            rooms (list[list[int]]): each item in this list represents a room, by the indices of the cells (given in row major order starting from 0)
            room_products (list[int]): the products corresponding to each room in rooms
        }
        output_board (list[list[int]]): output board which is to be verified

    Returns:
        dict[str, Union[Bool, str]]: {
            'result': Boolean indicating whether output_board is a valid solution,
            'reason': Reason of Invalidity, None otherwise
        }
    """
    input_board, rooms, room_products = input_sample['input_board'], input_sample['rooms'], input_sample['room_products']
    assert(len(room_products) == len(rooms))
    
    input_board = np.array(input_board)
    output_board = np.array(output_board)
    
    if input_board.shape != output_board.shape:
        return {
            'result': False,
            'reason': "Input and Output Boards do not have the same shape"
        }
    
    N, M = input_board.shape
    
    for i in range(N):    
        for j in range(M):
            if str(input_board[i,j]) != '0':
                if str(input_board[i][j]) != str(output_board[i][j]):
                    return {
                        'result': False,
                        'reason': f"Original numbers on Non-Empty cells of Input have been removed from the output board. Eg Cell ({i}, {j}) (0-indexed)"
                    }

    for i in range(N):
        ## row check
        elements_in_range = np.all(np.isin(output_board[i], np.arange(1,M+1)))
        unique_elements = len(np.unique(output_board[i]))
        if not elements_in_range or unique_elements != M:
            return {
                'result': False,
                'reason': f"Row should have each element from 1 to {M} exactly once. This constraint is violated for Row:{i} (0-indexed) of the output board"
            }
    
    for i in range(M):
        ## col check 
        elements_in_range = np.all(np.isin(output_board[:, i], np.arange(1, N+1)))
        unique_elements = len(np.unique(output_board[:, i]))
        if not elements_in_range or unique_elements != N:
            return {
                'result': False,
                'reason': f"Columns should have each element from 1 to {N} exactly once. This constraint is violated for Column: {i} (0-indexed) of the output board"
            }
    
    cells_covered = set()
    for room, room_product in zip(rooms, room_products):
        output_product = 1
        for cell in room:
            cells_covered.add(cell)
            x, y = cell//M, cell%M
            output_product *= output_board[x][y]
        if output_product != room_product:
            return {
                'result': False,
                'reason': f"Elements in a room should have a product equal to the room product. This constraint is violated for {room} which should have product={room_product}"
            }

    assert(len(cells_covered) == N*M)
    
    return {
        'result': True,
        'reason': None
    }

def MyVerifier():
    return InshiNoHeyaVerifier

if __name__ == "__main__":
    print(InshiNoHeyaVerifier(input_sample=
                                {
                                 'input_board': [
                                            [3,0,0,5,0],
                                            [0,1,0,0,0],
                                            [5,0,0,1,0],
                                            [0,0,0,0,0],
                                            [0,0,0,0,0],
                                          ]
                              ,
                              'rooms': [
                                        [0,5],
                                        [1,2],
                                        [3],
                                        [4,9,14],
                                        [6,7],
                                        [8,13],
                                        [10,11],
                                        [12,17,22],
                                        [15,20],
                                        [16,21],
                                        [18,19],
                                        [23,24]
                                    ],
                              'room_products': [
                                      6,
                                      4,
                                      5,
                                      40,
                                      3,
                                      4,
                                      15,
                                      40,
                                      4,
                                      10,
                                      6,
                                      3
                              ]},
                              output_board=[
                                            [3,4,1,5,2],
                                            [2,1,3,4,5],
                                            [5,3,2,1,4],
                                            [1,5,4,2,3],
                                            [4,2,5,3,1],
                                          ]
                              ))

    print(InshiNoHeyaVerifier(input_sample=
                              {
                                  'input_board': [
                                            [3,0,0,0,4],
                                            [0,5,0,1,0],
                                            [0,0,0,0,0],
                                            [0,2,4,3,0],
                                            [2,0,0,0,5],
                                          ]
                              ,
                              'rooms': [
                                        [0,1],
                                        [2,3],
                                        [4,9],
                                        [5,6,7],
                                        [8,13,18],
                                        [10,11],
                                        [12,17,22],
                                        [14,19],
                                        [15,20],
                                        [16,21],
                                        [23,24]
                                    ],
                              'room_products': [
                                      3,
                                      10,
                                      8,
                                      60,
                                      15,
                                      4,
                                      8,
                                      3,
                                      10,
                                      6,
                                      20
                              ]},
                              output_board=[
                                            [3,1,5,2,4],
                                            [4,5,3,1,2],
                                            [1,4,2,5,3],
                                            [5,2,4,3,1],
                                            [2,3,1,4,5],
                                          ]
                              ))
    