from z3 import *


def JobShopSolver(input_sample, **kwargs):
    num_machines = input_sample['num_machines']
    num_jobs= input_sample['num_jobs']
    T = input_sample['T']
    order_of_processing= input_sample['order_of_processing']
    processing_times= input_sample['processing_times']
    
    # Create solver instance
    s = Solver()

    # Start times for each task of each job on each machine, in order of processing
    start_times = [[Int(f'start_{j}_{m}') for m in range(num_machines)] for j in range(num_jobs)]

    # End times for makespan calculation
    end_times = [Int(f'end_{j}') for j in range(num_jobs)]

    # Constraint: Each task starts only after the previous one in the job has completed
    for j in range(num_jobs):
        s.add(start_times[j][0] >= 0)
        for m in range(1, num_machines):
            s.add(start_times[j][m] >= start_times[j][m-1] + processing_times[j][m-1])
            s.add(start_times[j][m] >= 0)


    # Constraint: No overlap on each machine
    for m in range(1, num_machines + 1):
        for j in range(num_jobs):
            for k in range(j + 1, num_jobs):
                # Find which job uses the machine m
                job_j_machine = order_of_processing[j].index(m)
                job_k_machine = order_of_processing[k].index(m)
                s.add(Or(start_times[j][job_j_machine] + processing_times[j][job_j_machine] <= start_times[k][job_k_machine],
                        start_times[k][job_k_machine] + processing_times[k][job_k_machine] <= start_times[j][job_j_machine]))

    # Makespan is the maximum end time among all jobs
    makespan = Int('makespan')
    # Constraint: Compute the end time of each job
    for j in range(num_jobs):
        last_machine_index = num_machines - 1
        s.add(end_times[j] == start_times[j][last_machine_index] + processing_times[j][last_machine_index])
        s.add(makespan >= end_times[j])

    # Constraint: Makespan should be less than or equal to T
    s.add(makespan <= T)

    # Check if there is a solution
    if s.check() == sat:
        # m = s.model()
        # for j in range(num_jobs):
            # print(f"Job {j}: Start times on machines =", [m[start_times[j][mi]].as_long() for mi in range(num_machines)])
        return ["YES"]
    else:
       return ["NO"]
    
def MySolver():
    return JobShopSolver    

if __name__ == "__main__":
    # Input data
    num_machines = 2
    num_jobs = 2
    order_of_processing = [
        [1, 2],
        [2, 1]
    ]
    processing_times = [
        [10, 5],
        [11, 4]
    ]
    T = 15  # maximum makespan we are checking against
    
    input_sample = {
        'num_machines': num_machines,
        'num_jobs': num_jobs,
        'T': T,
        'order_of_processing': order_of_processing,
        'processing_times': processing_times
    }
    print(JobShopSolver(input_sample))
    

