from z3 import *

def KeisukeSolver(input_sample, **kwargs):
    input_board, across_nums, down_nums = input_sample['input_board'], input_sample['across_numbers'], input_sample['down_numbers']
    # Read input from the file
    m, n = len(input_board), len(input_board[0])
    for i in range(m):
        for j in range(n):
            input_board[i][j] = str(input_board[i][j])

    cells = [[Int('cell_%d_%d' % (i,j)) for j in range(n)] for i in range(m)]

    # Create a Z3 solver
    board_solver = Solver()

    # Constraints for cells that are either a dash or a number
    for i in range(m):
        for j in range(n):
            if input_board[i][j] == '0':
                board_solver.add(cells[i][j] >= 0, cells[i][j] <= 9)
            elif input_board[i][j] in '123456789':
                board_solver.add(cells[i][j] == int(input_board[i][j]))
            else:
                board_solver.add(cells[i][j] == -1)

    # Helper function to add constraints for numbers
    def add_constraints_for_numbers(nums, is_across):
        for num in nums:
            str_num = str(num)
            len_num = len(str_num)

            possible_places = []

            # Check for each possible place where the number can appear
            for i in range(m):
                for j in range(n):
                    if is_across:
                        end_j = j + len_num
                        if end_j <= n:
                            seq = cells[i][j:end_j]
                            if '-' not in [input_board[i][k] for k in range(j, end_j)]:
                                possible_places.append(seq)
                    else:
                        end_i = i + len_num
                        if end_i <= m:
                            seq = [cells[k][j] for k in range(i, end_i)]
                            if '-' not in [input_board[k][j] for k in range(i, end_i)]:
                                possible_places.append(seq)

            # Adding the constraint for the number to appear in one of the possible places
            or_constraints = []
            for seq in possible_places:
                and_constraints = [seq[idx] == int(str_num[idx]) for idx in range(len_num)]
                or_constraints.append(And(and_constraints))
            
            board_solver.add(Or(or_constraints))

    # Add constraints for numbers
    add_constraints_for_numbers(across_nums, True)
    add_constraints_for_numbers(down_nums, False)

    # Solve the problem
    if board_solver.check() == sat:
        model = board_solver.model()
        solution_board = []
        for i in range(m):
            row = []
            for j in range(n):
                if model[cells[i][j]] == -1:
                    row.append("-")
                else:
                    row.append(int(str(model[cells[i][j]])))
            solution_board.append(row)
        return [solution_board] ## return one of the possible solutions
    else:
        print("Problem is unsatisfiable!")
        return [None]

def MySolver():
    return KeisukeSolver

if __name__ == "__main__":
    input_sample = {        
        'input_board': [[0,0,'-',0,0], 
                    ['-',0,0,0,0], 
                    [0,0,0,'-',0], 
                    [0,0,0,0,0], 
                    ['-',0,'-',0,'-']],
        'across_numbers': [13, 23, 233, 3221, 21222],
        'down_numbers': [12, 21, 22, 232, 3132, 33313]
    }
    solved_board = KeisukeSolver(input_sample)
    print(solved_board)

    input_sample = {
        'input_board': [[0,0,'-',0,0], 
                        [0,0,0,0,0], 
                        [0,3,4,0,0], 
                        [0,'-',0,0,0], 
                        [0,5,'-',0,1]],
        'across_numbers': [25, 15130, 205, 33415],
        'down_numbers': [53, 142, 50551, 21302]
    }
    solved_board = KeisukeSolver(input_sample)
    print(solved_board)