 import numpy as np
from functools import reduce

def KenKenVerifier(input_sample, output_board):
    input_board, cages, operations, targets = input_sample['input_board'], input_sample['subgrids'], input_sample['operations'], input_sample['targets']
    input_board = np.array(input_board)
    output_board = np.array(output_board)
    if input_board.shape != output_board.shape:
        return {
            'result': False,
            'reason': "Input and Output Board shape must match"
        }
    
    assert (input_board.shape[0] == input_board.shape[1])
    N = input_board.shape[0]
    
    for i in range(N):
        for j in range(N):
            if input_board[i, j] != 0 and output_board[i, j] != input_board[i, j]:
                return {
                    'result': False,
                    'reason': f'Original Numbers on Non-Empty cells of the input board removed from the output board. Eg Cell ({i}, {j}) (0-indexed)'
                }
    
    valid_elements = np.all(np.isin(output_board.flatten(), np.arange(1, N+1)))
    if not valid_elements:
        return {
            'result': False,
            'reason': f"Output board must only contain numbers from 1 to {N}"
        }

    ### row constraints
    for i in range(N):
        num_unique_elements = len(np.unique(output_board[i, :]))
        if num_unique_elements != N:
            return {
                'result': False,
                'reason': f"Each row of the output board must contain numbers from 1 to {N} exactly once. This constraint is violated for Row: {i} (0-indexed) of the output board"
            }

    ### column constraints
    for i in range(N):
        num_unique_elements = len(np.unique(output_board[:, i]))
        if num_unique_elements != N:
            return {
                'result': False,
                'reason': f'Each column of the output board must contain numbers from 1 to {N} exactly once. This constraint is violated for Column: {i} (0-indexed) of the output board'
            }
    
    for cage, operation, target in zip(cages, operations, targets):
        verification_result = True
        if operation == '?':
            assert len(cage) == 1
            x, y = cage[0]//N, cage[0]%N
            if output_board[x, y] != target:
                verification_result = False
        elif operation == '%' or operation == '-':
            if operation == '%':
                operation = '//'
            assert len(cage) == 2
            x1, y1 = cage[0]//N, cage[0]%N
            x2, y2 = cage[1]//N, cage[1]%N
            element_1, element_2 = output_board[x1, y1], output_board[x2, y2]
            result_1 = eval(f"{element_1} {operation} {element_2} == {target}")
            result_2 = eval(f"{element_2} {operation} {element_1} == {target}")
            if not result_1 and not result_2:
                verification_result = False
        else:
            assert operation == "*" or operation == "+"
            numbers = [output_board[ele//N][ele%N] for ele in cage]
            if operation == "*":
                result = reduce(lambda x, y: x*y, numbers)
            elif operation == "+":
                result = reduce(lambda x, y: x+y, numbers)
            if result != target:
                verification_result = False
        if not verification_result:
            return {
                'result': False,
                'reason': f"Cage constraints not satisfied for cage: {cage} with operation: {operation} and target: {target}"
            }
    
    return {
        'result': True,
        'reason': None
    }

def MyVerifier():
    return KenKenVerifier
            
if __name__ == "__main__":
    print(KenKenVerifier(input_sample={
                            'input_board':[
                                      [1,0,0],
                                      [3,0,0],
                                      [0,0,0]
                                     ]
                            ,
                            'subgrids': [
                             [0, 1],
                             [2],
                             [3],
                             [4,5,8],
                             [6,7]
                         ],
                         'operations':
                         [
                            '+',
                            '?',
                            '?',
                            '+',
                            '+'
                         ],
                         'targets':
                         [
                            3,
                            3,
                            3,
                            4,
                            5
                         ]},
                         output_board=[
                                      [1,2,3],
                                      [3,1,2],
                                      [2,3,1]
                                     ]
                         ))


    print(KenKenVerifier(input_sample={
                            'input_board': [
                                      [0,2,0,0],
                                      [0,1,0,0],
                                      [3,0,2,0],
                                      [0,3,0,0],
                                     ]
                         ,
                         'subgrids': [
                             [0, 1],
                             [2, 6],
                             [3],
                             [4,8],
                             [5,9],
                             [7,11],
                             [10,14,15],
                             [12,13]
                         ],
                         'operations':
                         [
                            '%',
                            '+',
                            '?',
                            '-',
                            '-',
                            '-',
                            '*',
                            '-'
                         ],
                         'targets':
                         [
                            2,
                            7,
                            4,
                            1,
                            3,
                            2,
                            4,
                            1
                         ]},
                         output_board=[
                                      [1,2,3,4],
                                      [2,1,4,3],
                                      [3,4,2,1],
                                      [4,3,1,2],
                                     ]
                         ))
                
                
    