from z3 import *

def LongestPathSolver(input_sample, **kwargs):
    N = input_sample['number_of_nodes']
    edges = input_sample['edges']
    K = input_sample['k']
    
    ### we need to determine if a path exists with atleast K edges
    ### if no path exists with exactly K edges, then no path with more edges exists
    path_length = K
    solver = Solver()
    variables = [Int(f"x_{i}") for i in range(path_length+1)] ### k+1 nodes on a k-edge path
    
    ### initial constraints
    for i in range(path_length+1):
        solver.add(variables[i] >=0, variables[i] < N)
    solver.add(Distinct(*variables)) ### path should not have repeated vertices
    
    ### ensure that all adjacent vertices on the path are connected
    for idx in range(path_length):
        x, y = variables[idx], variables[idx+1]
        for i in range(N):
            for j in range(i+1, N):
                if (i, j) not in edges and (j, i) not in edges:
                    ### ensure contrapositive, that if an edge between two vertices does not exist then they cannot be adjacent
                    solver.add(Not(And(x == i, y == j)))
                    solver.add(Not(And(x == j, y == i)))
                        
    if solver.check() == sat:
        return ["YES"]
    else:
        return ["NO"]

def MySolver():
    return LongestPathSolver

if __name__ == "__main__":
    input_sample = {
        'number_of_nodes': 4,
        'edges' : [(0,1), (0,2), (0,3)],
        'k': 2
    } ## YES
    print(LongestPathSolver(input_sample=input_sample))
    
    input_sample = {
        'number_of_nodes': 4,
        'edges' : [(0,1), (0,2), (0,3)],
        'k': 3
    } ## NO
    print(LongestPathSolver(input_sample=input_sample))

    input_sample = {
        'number_of_nodes': 6,
        'edges' : [(0,1), (1, 2), (2, 3), (2, 4), (4, 5)],
        'k': 5
    } ## NO
    print(LongestPathSolver(input_sample=input_sample))

    input_sample = {
        'number_of_nodes': 6,
        'edges' : [(0,1), (1, 2), (2, 3), (2, 4), (4, 5)],
        'k': 4
    } ## YES
    print(LongestPathSolver(input_sample=input_sample))

    input_sample = {
        'number_of_nodes': 4,
        'edges' : [(0,1), (2, 3)],
        'k': 2
    } ## NO
    print(LongestPathSolver(input_sample=input_sample))

    input_sample = {
        'number_of_nodes': 4,
        'edges' : [(0,1), (2, 3)],
        'k': 1
    } ## YES
    print(LongestPathSolver(input_sample=input_sample))