### reference: https://stackoverflow.com/questions/78234385/z3-to-solve-a-puzzle8-blocks-tiles-please
from z3 import *


Direction, (Up, Down, Left, Right) = EnumSort('Direction', ('Up', 'Down', 'Left', 'Right'))

class Grid:
    def __init__(self, board = None):
        self.board = board

        # Find the location of zero if a board is given
        if board:
           N = len(board)
           zeroLoc = [e for row in board for e in row].index(0)
           self.x  = zeroLoc % N
           self.y  = zeroLoc //  N
    
    def __len__(self):
        if self.board:
            return len(self.board)
        else:
            raise Exception('Empty Grid does not have a length')

# Pick a symbolic location from the grid
def pick(grid, x, y):
    val = 0
    N = len(grid)
    for row in range(N):
        for col in range(N):
            val = If(And(row == y, col == x), grid.board[row][col], val)
    return val

def reshape_list(flat_list, n):
    return [flat_list[i * n:(i + 1) * n] for i in range(n)]

# Move in a particular direction, if possible
def move(goodSofar, direction, grid):
    N = len(grid)
    invalid = Or( And(direction == Up,    grid.y <= 0)
                , And(direction == Down,  grid.y >= N-1)
                , And(direction == Left,  grid.x <= 0)
                , And(direction == Right, grid.x >= N-1)
                , Not(goodSofar)
                )

    newGrid   = Grid()
    newGrid.x = If(invalid, grid.x, If(direction == Left, grid.x - 1, If(direction == Right, grid.x + 1, grid.x)))
    newGrid.y = If(invalid, grid.y, If(direction == Up,   grid.y - 1, If(direction == Down,  grid.y + 1, grid.y)))
    newVal    = pick(grid, newGrid.x, newGrid.y)

    newBoard  = [If(invalid, e, If(And(grid.x    == c, grid.y    == r), newVal,
                                If(And(newGrid.x == c, newGrid.y == r), 0, e)))
                 for (r, row) in enumerate(grid.board) for (c, e) in enumerate(row)]

    newGrid.board = reshape_list(newBoard, N)

    return (Not(invalid), newGrid)


def SlidingTileSolver(input_sample, **kwargs):
    initBoard = input_sample['initial_board']
    finalBoard = input_sample['final_board']
    k = input_sample['k']
    
    for moves in range(k+1):
        symMoves = [Const(f"m_{i}", Direction) for i in range(moves)]
        newBoard = Grid(initBoard)
        good     = True
        for d in symMoves:
            (valid, newBoard) = move(good, d, newBoard)
            good              = And(good, valid)

        s = Solver()
        s.add(good)
        for (req, got) in zip([e for row in finalBoard for e in row], [e for row in newBoard.board for e in row]):
            s.add(req == got)

        r = s.check()

        if r == unsat: continue

        if r == sat:
           return ["YES"]
    return ['NO']
        
def MySolver():
    return SlidingTileSolver

if __name__ == "__main__":

    initBoard = [[8,7,6],[5,4,3],[2,1,0]]
    finalBoard = [[0,8,7],[5,4,6],[2,1,3]]

    print(SlidingTileSolver(
        input_sample={
            'initial_board': initBoard,
            'final_board': finalBoard,
            'k': 5
        }
    ))
    
    initBoard = [[1, 0], [3, 2]]
    finalBoard = [[1, 2], [0, 3]]

    print(SlidingTileSolver(
        input_sample={
            'initial_board': initBoard,
            'final_board': finalBoard,
            'k': 2
        }
    ))