import re
from collections import Counter

def SummleVerifier(input_sample, output_sample, **kwargs):
    n = input_sample["n"] ## number of equations
    operators = input_sample["operators"] ## operators to be used
    target = input_sample["target"] ## final target, last result
    numbers  = input_sample["numbers"] ## numbers to be used
    
    numbers_to_be_used = Counter(numbers)
    operators_to_be_used = Counter(operators)
    
    if len(output_sample) != n:
        return {
            "result": False,
            "reason": f"The output should have {n} equations"
        }
    
    pattern = re.compile(r'^(-?\d+)\s*([+*/-])\s*(-?\d+)\s*=\s*(-?\d+)$')
    for equation_idx, equation in enumerate(output_sample):
        match = pattern.match(equation)
        if not match:
            return {
                "result": False,
                "reason": "Each Equation in the output should be of the form OPERAND1 OPERATOR OPERAND2 = RESULT"
            }
        
        # Extract components from the regex match
        operand1, operator, operand2, expected_result = match.groups()

        # Convert strings to integers
        operand1 = int(operand1)
        operand2 = int(operand2)
        expected_result = int(expected_result)
        
        if operator not in operators_to_be_used or operators_to_be_used[operator] == 0:
            return {
                "result": False,
                "reason": "Only operator provided should be used and each can be used once"
            }
        operators_to_be_used[operator] -= 1
        
        if operand1 not in numbers_to_be_used or numbers_to_be_used[operand1] == 0:
            return {
                "result": False,
                "reason": "Only numbers provided and intermediate results can be used and each can be used once"
            }
        numbers_to_be_used[operand1] -= 1

        if operand2 not in numbers_to_be_used or numbers_to_be_used[operand2] == 0:
            return {
                "result": False,
                "reason": "Only numbers provided and intermediate results can be used and each can be used once"
            }
        
        numbers_to_be_used[operand2] -= 1
        
        
        # Calculate the result based on the operator
        try:
            if operator == '+':
                result = operand1 + operand2
            elif operator == '-':
                result = operand1 - operand2
            elif operator == '*':
                result = operand1 * operand2
            elif operator == '/':
                if operand2 == 0:
                    return {
                        "result": False,
                        "reason": "Division by zero"
                    }
                result = operand1 / operand2
            else:
                return {
                    "result": False,
                    "reason": "Unsupported operator"
                }
        except Exception as e:
            return {
                "result": False,
                "reason": e
            }

        # Validate the result against the expected result
        if result != expected_result:
            return {
                "result": False,
                "reason": f"{equation} is invalid"
            }
            
        if result not in numbers_to_be_used:
            numbers_to_be_used[result] = 0
        numbers_to_be_used[result] += 1
        
        if equation_idx == n-1:
            if expected_result != target:
                return {
                    "result": False,
                    "reason": "The last result must be same as the target"
                }
    return {
        "result": True,
        "reason": None
    }
    
def MyVerifier():
    return SummleVerifier


if __name__ == "__main__":
    input_sample_1 = {
        'numbers' : [4, 3, 5],
        'target' : 35,
        'n' : 2,
        'operators' : ['+', '*']
    }
    output_sample_1 =[ '3 + 4 = 7', '5 * 7 = 35']
    solution = SummleVerifier(input_sample_1, output_sample_1)
    print("Solution:", solution)

    input_sample_2 = {
        'numbers' : [4, 3, 5],
        'target' : 35,
        'n' : 2,
        'operators' : ['+', '*']
    }
    output_sample_2 =[ '5 + 10 = 15', '15 + 20 = 35']
    solution = SummleVerifier(input_sample_2, output_sample_2)
    print("Solution:", solution)
    

        
        
    