import numpy as np
import math


def SymmetricSudokuVerifier(input_board, output_board, **kwargs):
    subgrid_size = int(math.sqrt(len(output_board)))
    input_board = np.array(input_board)
    output_board = np.array(output_board)

    if (input_board.shape) != (output_board.shape):
        return {
            "result": False,
            "reason": "Input and Output Boards do not have the same shape",
        }
    for i in range(len(input_board)):
        for j in range(len(input_board[0])):
            if str(input_board[i][j]) != "0":
                if str(input_board[i][j]) != str(output_board[i][j]):
                    return {
                        "result": False,
                        "reason": f"Original numbers on non-empty cells have been removed, For eg. At Cell ({i}, {j}) (0-indexed)",
                    }

    for idx, row in enumerate(output_board):
        if len(row) != len(set(row)):
            return {
                "result": False,
                "reason": f"Unique elements in row constraint is violated for Row: {idx} (0-indexed) of the Output Board",
            }

    # Check columns
    num_rows = len(output_board)
    num_cols = len(output_board[0])
    for j in range(num_cols):
        column = [output_board[i][j] for i in range(num_rows)]
        if len(column) != len(set(column)):
            return {
                "result": False,
                "reason": f"Unique elements in column constraint is violated for Column: {j} (0-indexed) of the Output Board",
            }

    subgrids = []
    for i in range(0, len(output_board), subgrid_size):
        for j in range(0, len(output_board), subgrid_size):
            subgrid = [
                row[j : j + subgrid_size] for row in output_board[i : i + subgrid_size]
            ]
            subgrids.append(subgrid)
    subgrids = np.array(subgrids)
    for subgrid in subgrids:
        transpose_subgrid = subgrid.T
        if np.array_equal(transpose_subgrid, subgrid):
            continue
        else:
            return {
                "result": False,
                "reason": f"Symmetric constraint of {subgrid_size}x{subgrid_size} subgrids in {subgrid} is violated",
            }

    return {"result": True, "reason": None}


def MyVerifier():
    return SymmetricSudokuVerifier


if __name__ == "__main__":
    input_board = [[4, 0, 0, 2], [0, 0, 0, 0], [0, 0, 0, 4], [3, 2, 4, 1]]

    output_board = [[4, 1, 3, 2], [1, 4, 2, 3], [2, 3, 1, 4], [3, 2, 4, 1]]

    print(SymmetricSudokuVerifier(input_board, output_board))





