import re
import subprocess
import math

def clean_lines(lines: list[str]) -> list[str]:
    return [line.strip() for line in lines if line.strip()]

def input_from_text(input_text: list[str], **kwargs):
    input_text = clean_lines(input_text)
    input_board = [list(map(int, line.strip().split())) for line in input_text]
    return input_board

def output_from_text(output_text: list[str], **kwargs):
    try:
        output_text = clean_lines(output_text)
        num_lines = len(output_text)
        if num_lines == 0:
            return {
                    "OUTPUT": None,
                    "ERROR": "Output is Empty"
            }
        output_board = [list(map(int, line.strip().split())) for line in output_text]
        for row in output_board:
            if len(row) != num_lines:
                return {
                    "OUTPUT": None,
                    "ERROR": f"Each non empty row in output.txt must have {num_lines} numbers"
                }
        return {
            'OUTPUT': output_board,
            "ERROR": None
        }
    except Exception as e:
        return {
            "OUTPUT": None,
            "ERROR": f"Could not parse output.txt, output format should match description, when my script tried to parse output.txt the following exception was generated:\n{e}"
        }

def input_to_text_string(input_sample, **kwargs) -> str:
    return "\n".join(" ".join(map(str, row)) for row in input_sample)

def output_to_text_string(output_sample, **kwargs) -> str:
    return "\n".join(" ".join(map(str, row)) for row in output_sample)

with open('./rules/rules.txt', 'r') as file:
    rules = file.read()

with open('./rules/input-format.txt', 'r') as file:
    input_format = file.read()

with open('./rules/output-format.txt', 'r') as file:
    ouptut_format = file.read()


def get_puzzle_lm_prompt():
    return"""

The Python Program is expected to read the input from input.txt and convert that particular input instance to corresponding constraints, which it should pass to the Z3 solver, and then it should use the Z3 solver's output to write the solution to a file named output.txt
Don't write anything apart from the Python Program; use Python comments if needed.
Write the python code enclosed in ```python ```
"""

def get_prompt(sample_input, sample_output, mode='pal-sat', **kwargs) -> str:
    return f"""
Write a Python Program to solve the following problem.

Task:
{rules}

Input Format:
{input_format}
Sample Input:
{input_to_text_string(input_sample=sample_input)}

Output Format:
{ouptut_format}
Sample Output:
{output_to_text_string(output_sample=sample_output)}

{get_puzzle_lm_prompt()}
    """

def clean_code_block(block):
    return re.sub(r'^\s*python', '', block, flags=re.IGNORECASE).strip()


def parse_gpt_output(gpt_output, depth):
    code_blocks = re.findall(r'```(.*?)```', gpt_output, re.DOTALL)

    if not code_blocks:
        print(f"Warning at Depth={depth}: No Code Blocks: Picking Entire Text")
        return gpt_output

    # If there's only one code block, return it after cleaning
    if len(code_blocks) == 1:
        return clean_code_block(code_blocks[0])
    
    # If multiple code blocks, prioritize one starting with 'python'
    for block in code_blocks:
        if block.strip().lower().startswith('python'):
            print(f"Info at Depth={depth}: Multiple Code Blocks - Prioritizing the 'python' block")
            return clean_code_block(block)
    
    # If none start with 'python', just pick the first
    print(f"Warning at Depth={depth}: Multiple Code Blocks in Output, Picking First Code Block")
    return clean_code_block(code_blocks[0])

def run_python_file(filename, depth, timeout):
    try:
        # Run the Python file as a subprocess
        result = subprocess.run(['python3', filename], stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True, timeout=timeout)

        # Get stdout and stderr outputs
        stdout_output = result.stdout
        stderr_output = result.stderr

        return {
            "STD-OUTPUT": str(stdout_output),
            "STD-ERROR": str(stderr_output)
        }

    except subprocess.CalledProcessError as e:
        # Handle any exceptions or errors
        print(f"Error occurred at Depth-{depth}:")
        return {
            "RUNTIME-ERROR": str(e)
        }

    except subprocess.TimeoutExpired:
        # Handle timeout exception
        print(f"Timeout occurred at Depth-{depth}. The script generated took too long to execute.")
        return {
            "TIMEOUT-ERROR": f"Your code was ineffecient and took more than {timeout} seconds to execute"
        }



def get_runtime_error_iterative_prompt(feedback_level, input_sample, runtime_error, **kwargs)->str:
    if feedback_level == 0:
        return "Your code is incorrect, rewrite your code and fix the mistake."
    elif feedback_level == 1:
        return "Your code is incorrect and produces a runtime error, rewrite your code and fix the mistake."
    elif feedback_level == 2:
        return f"Your code is incorrect and produces a runtime error, for the following input:\n{input_to_text_string(input_sample, **kwargs)}\n rewrite your code and fix the mistake"
    else:
        return f"Your code is incorrect and produces the following runtime error:\n{runtime_error}\n for the following input:\n{input_to_text_string(input_sample, **kwargs)}\nrewrite your code and fix the mistake. Only write Python code and nothing else, also write the python code enclosed in ```python ```"


def get_verification_error_iterative_prompt(feedback_level, input_sample, output_sample, error_type, error, generated_output, **kwargs) -> str:
    if feedback_level == 0:
        return "Your code is incorrect, rewrite your code and fix the mistake."
    elif feedback_level == 1:
        return f"Your code is incorrect ({error_type}), rewrite your code and fix the mistake."
    elif feedback_level == 2:
        return f"Your code is incorrect produces the following kind of mistake: {error_type}, when run on the input:\n{input_to_text_string(input_sample, **kwargs)}\n rewrite your code and fix the mistake."
    elif feedback_level == 3:
        return f"Your code is incorrect produces the following kind of mistake: {error_type}, when run on the input:\n{input_to_text_string(input_sample, **kwargs)}\n for which your code produces the following output:\n{generated_output}\n rewrite your code and fix the mistake."
    elif feedback_level == 4:
        return f"Your code is incorrect produces the following kind of mistake: {error_type}, when run on the input:\n{input_to_text_string(input_sample, **kwargs)}\n for which one of the correct output is:\n{output_to_text_string(output_sample, **kwargs)}\n and for which your code produces the following output:\n{generated_output}\n rewrite your code and fix the mistake. Only write Python code and nothing else, also write the python code enclosed in ```python ```"
    elif feedback_level == 5:
        return f"Your code is incorrect produces the following kind of mistake: {error_type}, {error} when run on the input:\n{input_to_text_string(input_sample, **kwargs)}\n rewrite your code and fix the mistake."
    elif feedback_level == 6:
        return f"Your code is incorrect produces the following kind of mistake: {error_type}, {error} when run on the input:\n{input_to_text_string(input_sample, **kwargs)}\n for which one of the correct output is:\n{output_to_text_string(output_sample, **kwargs)}\n rewrite your code and fix the mistake."
    else:
        return f"Your code is incorrect produces the following kind of mistake: {error_type}, {error} when run on the input:\n{input_to_text_string(input_sample, **kwargs)}\n for which one of the correct output is:\n{output_to_text_string(output_sample, **kwargs)}\n and your code generated:\n{generated_output} \nrewrite your code and fix the mistake."

def get_timeout_error_iterative_prompt(feedback_level, input_sample, timeout_error, **kwargs) -> str:
    if feedback_level == 0:
        return "Your code is slow, for certain inputs, rewrite your code and optimize it"
    else:
        return f"Your code ran too slowly for the input:\n{input_to_text_string(input_sample, **kwargs)}\nrewrite your code and optimize it. Only write Python code and nothing else, also write the python code enclosed in ```python ```"