import numpy as np
import math


def SudokuVerifier(input_board, output_board, **kwargs):
    subgrid_size = int(math.sqrt(len(output_board)))
    input_board = np.array(input_board)
    output_board = np.array(output_board)

    if (input_board.shape) != (output_board.shape):
        return {
            "result": False,
            "reason": "Input and Output Boards do not have the same shape",
        }
    for i in range(len(input_board)):
        for j in range(len(input_board[0])):
            if str(input_board[i][j]) != "0":
                if str(input_board[i][j]) != str(output_board[i][j]):
                    return {
                        "result": False,
                        "reason": f"Original numbers on Non-Empty cells have been removed in output. Eg. Cell ({i+1}, {j+1}) (1-indexed)",
                    }

    for idx, row in enumerate(output_board):
        if len(row) != len(set(row)):
            return {
                "result": False,
                "reason": f"Unique elements in row constraint is violated for Row: {idx} (0-indexed) of the output board",
            }

    # Check columns
    num_rows = len(output_board)
    num_cols = len(output_board[0])
    for j in range(num_cols):
        column = [output_board[i][j] for i in range(num_rows)]
        if len(column) != len(set(column)):
            return {
                "result": False,
                "reason": f"Inique elements in column constraint is violated for Column: {j} (0-indexed) of the output board",
            }

    subgrids = []
    for i in range(0, len(output_board), subgrid_size):
        for j in range(0, len(output_board), subgrid_size):
            subgrid = [
                row[j : j + subgrid_size] for row in output_board[i : i + subgrid_size]
            ]
            subgrids.append(subgrid)

    for subgrid in subgrids:
        nums = set()
        for i in range(len(subgrid)):
            for j in range(len(subgrid)):
                if subgrid[i][j] not in nums:
                    nums.add(subgrid[i][j])
                else:
                    return {
                        "result": False,
                        "reason": f"Uniqueness of elements in {subgrid_size}x{subgrid_size} subgrid violated for the following: {subgrid} of the output.",
                    }

    return {"result": True, "reason": None}


def MyVerifier():
    return SudokuVerifier


if __name__ == "__main__":
    input_board = [[0, 2, 3, 1], [3, 0, 4, 2], [0, 0, 1, 3], [0, 0, 0, 4]]

    output_board = [[4, 2, 3, 1], [3, 1, 4, 2], [2, 4, 1, 3], [1, 3, 2, 4]]

    print(SudokuVerifier(input_board, output_board))
