import os
import torch
import torchvision as tv
from torch import optim, nn
from torch.utils.data import DataLoader

import cv2
import sys
sys.path.append('./')
from config import base_config
from dataset.openlane_data_2d import LaneDataset
from torchsummary import summary
import torch.nn.functional as F
from utils.utils import ego2image_torch, apply_mask_with_numpy, apply_mask_to_4d_array, apply_embedding_to_4d_array
from einops import rearrange, reduce
from einops.layers.torch import Rearrange
import numpy as np 
import torch.distributed as dist

class Custom_L1_Loss(nn.Module):
    def __init__(self):
        super(Custom_L1_Loss, self).__init__()
    
    def forward(self, input, target):
        diff = torch.abs(input - target)
        diff[diff < 0.5] = 0
        loss = diff.mean()
        return loss

class Motion_loss(nn.Module):
    def __init__(self):
        super(Motion_loss, self).__init__()
    def forward(self, input, target):
        ##
        # input: B C N, C:[x y z class]
        # target: B C N, C:[x y z class]
        zero_torch = torch.zeros(3, 1).to(input.device)
        total_loss = torch.tensor(0.0).to(input.device)
        for b in range(input.shape[0]):
            max_num_class = torch.max(input[b, 3, :])
            max_num_class = int(max_num_class)
            for i in range(max_num_class):
                mask = input[b, 3, :] == i+1
                input_masked = torch.where(mask.unsqueeze(0), input[b, :3, :], zero_torch)
                target_masked = torch.where(mask.unsqueeze(0), target[b, :3, :], zero_torch)

                # 过滤掉 zero_torch 的列
                valid_indices = (input_masked != 0).any(dim=0)
                valid_indices_target = (target_masked != 0).any(dim=0)
                lane_pre = input_masked[:, valid_indices]
                lane_gt = target_masked[:, valid_indices_target]
                if lane_pre.shape[1] < 2:
                    loss = torch.abs(lane_pre - lane_gt).mean()
                    total_loss = total_loss + loss
                else:
                    dis_lane = lane_pre - lane_gt
                    dis_lane_f = dis_lane[:, 1:]
                    dis_lane_b = dis_lane[:, :-1]
                    # loss
                    cross = torch.cross(dis_lane_f, dis_lane_b, dim=0)
                    loss = torch.abs(cross).mean()
                    total_loss = total_loss + loss
                    if torch.isnan(loss):
                        print(f"NaN detected! lane_pre shape: {lane_pre.shape}, lane_tgt shape: {lane_gt.shape}")
                        assert not torch.isnan(loss), "Loss contains NaN values"
        return total_loss

class Residual(nn.Module):
    def __init__(self, module, downsample=None):
        super(Residual, self).__init__()
        self.module = module
        self.downsample = downsample
        self.relu = nn.ReLU(inplace=True)

    def forward(self, x):
        identity = x
        out = self.module(x)

        if self.downsample is not None:
            identity = self.downsample(x)

        out = out + identity
        return self.relu(out)
    
class DoubleConv(nn.Module):
    """(convolution => [BN] => ReLU) * 2"""

    def __init__(self, in_channels, out_channels, mid_channels=None):
        super().__init__()
        if not mid_channels:
            mid_channels = out_channels
        self.double_conv = Residual(
            module=nn.Sequential(
            nn.Conv2d(in_channels, mid_channels, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(mid_channels),
            nn.ReLU(inplace=True),
            nn.Conv2d(mid_channels, out_channels, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(out_channels),
            nn.ReLU(inplace=True)),
            downsample=nn.Sequential(
                nn.Conv2d(in_channels, out_channels, kernel_size=1, stride=1, bias=False))
        )

    def forward(self, x):
        return self.double_conv(x)

class Down(nn.Module):
    """Downscaling with maxpool then double conv"""

    def __init__(self, in_channels, out_channels):
        super().__init__()
        self.maxpool_conv = nn.Sequential(
            nn.MaxPool2d(2),
            DoubleConv(in_channels, out_channels)
        )

    def forward(self, x):
        return self.maxpool_conv(x)

class Up(nn.Module):
    """Upscaling then double conv"""

    def __init__(self, in_channels, out_channels, bilinear=True):
        super().__init__()

        # if bilinear, use the normal convolutions to reduce the number of channels
        if bilinear:
            self.up = nn.Upsample(scale_factor=2, mode='bilinear', align_corners=True)
            self.conv = DoubleConv(in_channels, out_channels, in_channels // 2)
        else:
            self.up = nn.ConvTranspose2d(in_channels, in_channels // 2, kernel_size=2, stride=2)
            self.conv = DoubleConv(in_channels, out_channels)

    def forward(self, x1, x2):
        x1 = self.up(x1).contiguous()
        # input is CHW
        diffY = x2.size()[2] - x1.size()[2]
        diffX = x2.size()[3] - x1.size()[3]

        x1 = F.pad(x1, [diffX // 2, diffX - diffX // 2,
                        diffY // 2, diffY - diffY // 2]).contiguous()
        x = torch.cat([x2, x1], dim=1).contiguous()
        return self.conv(x)

class OutConv(nn.Module):
    def __init__(self, in_channels, out_channels):
        super(OutConv, self).__init__()
        self.conv = nn.Sequential(
            nn.Conv2d(in_channels, in_channels, 3,1,1),
            nn.BatchNorm2d(in_channels),
            nn.ReLU(inplace=True),
            nn.Conv2d(in_channels, out_channels, 3,1,1)
        )

    def forward(self, x):
        return self.conv(x)
class APM(nn.Module):
    def __init__(self, image_featmap_size):
        super(APM, self).__init__()
        ic, ih, iw = image_featmap_size
        sc, sh, sw = image_featmap_size
        self.image_featmap_size = image_featmap_size
        self.space_featmap_size = image_featmap_size
        self.fc_transform = nn.Sequential(
            nn.Linear(ih * iw, sh * sw),
            nn.ReLU(),
            nn.Linear(sh * sw, sh * sw),
            nn.ReLU(),
            nn.Linear(sh * sw, sh * sw),
            nn.ReLU()
        )
    def forward(self, x):
        x = x.view(list(x.size()[:2]) + [self.image_featmap_size[1] * self.image_featmap_size[2], ])
        bev_view = self.fc_transform(x)  # 拿出一个视角
        bev_view = bev_view.view(list(bev_view.size()[:2]) + [self.space_featmap_size[1], self.space_featmap_size[2]])
        return bev_view


class UNet(nn.Module):
    def __init__(self, n_channels, n_classes, bilinear=False):
        super(UNet, self).__init__()
        self.n_channels = n_channels
        self.n_classes = n_classes
        self.bilinear = bilinear

        self.inc = (DoubleConv(n_channels, 64))
        self.down1 = (Down(64, 128))
        self.down2 = (Down(128, 256))
        self.down3 = (Down(256, 512))
        factor = 2 if bilinear else 1
        self.down4 = (Down(512, 1024 // factor))
        self.apm = APM((1024 // factor, 20, 30))
        self.up1 = (Up(1024, 512 // factor, bilinear))
        self.up2 = (Up(512, 256 // factor, bilinear))
        self.up3 = (Up(256, 128 // factor, bilinear))
        self.up4 = (Up(128, 64, bilinear))
        self.out1 = (OutConv(64, 1))
        self.out2 = (OutConv(64, 1))
        self.out3 = (OutConv(64, 1))

    def forward(self, x):
        x1 = self.inc(x)
        x2 = self.down1(x1)
        x3 = self.down2(x2)
        x4 = self.down3(x3)
        x5 = self.down4(x4)
        apm = self.apm(x5)
        x5 = x5 + apm
        x = self.up1(x5, x4)
        x = self.up2(x, x3)
        x = self.up3(x, x2)
        x = self.up4(x, x1)
        x_pre = self.out1(x)
        y_pre = self.out2(x)
        z_pre = self.out3(x)
        return x_pre, y_pre, z_pre

    def use_checkpointing(self):
        self.inc = torch.utils.checkpoint(self.inc)
        self.down1 = torch.utils.checkpoint(self.down1)
        self.down2 = torch.utils.checkpoint(self.down2)
        self.down3 = torch.utils.checkpoint(self.down3)
        self.down4 = torch.utils.checkpoint(self.down4)
        self.up1 = torch.utils.checkpoint(self.up1)
        self.up2 = torch.utils.checkpoint(self.up2)
        self.up3 = torch.utils.checkpoint(self.up3)
        self.up4 = torch.utils.checkpoint(self.up4)
        self.out1 = torch.utils.checkpoint(self.out1)
        self.out2 = torch.utils.checkpoint(self.out2)
        self.out3 = torch.utils.checkpoint(self.out3)

class neck(nn.Module):
    def __init__(self, in_channels, middle_channels, out_channels):
        super(neck, self).__init__()
        self.conv = nn.Sequential(
            nn.Conv2d(in_channels, middle_channels, 3, 1, 1),
            nn.BatchNorm2d(middle_channels),
            nn.ReLU(inplace=True),
            nn.Conv2d(middle_channels, out_channels, 3, 1, 1),
            nn.BatchNorm2d(out_channels),
            nn.ReLU(inplace=True)
        )
    def forward(self, x):
        return self.conv(x)

class test_model(nn.Module):
    def __init__(self, args):
        super(test_model, self).__init__()
        self.unet = UNet(6, 3)
        # self.neck = neck(1, 64, 3)
        self.is_train = args.is_train
        self.loss_x = nn.MSELoss()
        self.loss_y = nn.MSELoss()
        self.loss_z = nn.MSELoss()
        self.loss_2d = Custom_L1_Loss()
        self.loss_motion = Motion_loss()

        # self.loss_p = nn.L1Loss()
    def forward(self, x, gt_3d=None, p=None, gt_2d=None, mask1=None, mask2=None, embedding=None):
        height, width = x.shape[-2], x.shape[-1]
        # x_mask = x[:, :3, :, :].contiguous()
        # x_mask = apply_mask_to_4d_array(x_mask, mask1).contiguous()
        mask1 = mask1.unsqueeze(1).contiguous()
        # mask = self.neck(mask1)
        x = torch.cat([x, mask1], dim=1).contiguous()
        x_pre, y_pre, z_pre = self.unet(x)
        x_xyz = torch.cat([x_pre, y_pre, z_pre], dim=1).contiguous()
        # 对x_pre实现掩码
        # x_pre = apply_mask_to_4d_array(x_pre, mask).contiguous()
        if self.is_train:
            x_xyz_ = rearrange(x_xyz, 'b c h w -> b c (h w)').contiguous()
            x_xyz_ = ego2image_torch(x_xyz_, p).contiguous()
            x_uv = rearrange(x_xyz_, 'b c (h w) -> b c h w', h=height, w=width).contiguous()
            x_xyz = apply_mask_to_4d_array(x_xyz, mask1).contiguous()
            gt_uv = x[:, 3:5, :, :].contiguous()
            x_xyz_lane = apply_embedding_to_4d_array(x_xyz, embedding).contiguous()
            gt_3d_lane = apply_embedding_to_4d_array(gt_3d, embedding).contiguous()
            x_xyz_lane = rearrange(x_xyz_lane, 'b c h w -> b c (h w)').contiguous()
            gt_3d_lane = rearrange(gt_3d_lane, 'b c h w -> b c (h w)').contiguous()

            x_pre = x_xyz[:, 0,:,:].contiguous()
            y_pre = x_xyz[:, 1,:,:].contiguous()
            z_pre = x_xyz[:, 2,:,:].contiguous()
            x_gt = gt_3d[:, 0,:,:].contiguous()
            y_gt = gt_3d[:, 1,:,:].contiguous()
            z_gt = gt_3d[:, 2,:,:].contiguous()

            loss_x = self.loss_x(x_pre, x_gt)*5e3 #5e3
            loss_y = self.loss_y(y_pre, y_gt)*1e2 #1e2
            loss_z = self.loss_z(z_pre, z_gt)*1.666e4 #1.666e4
            loss_motion = self.loss_motion(x_xyz_lane, gt_3d_lane)*1e-2 #1e-2
            loss_2d = self.loss_2d(x_uv, gt_uv)*1e-4 #1e-4
            return loss_x, loss_y, loss_z, loss_2d, loss_motion
        else:
            return x_xyz
