"""
Unified evaluation function dispatcher for WebArena Verified tasks.

This module provides a centralized registry for both built-in evaluation functions
and site-specific evaluation functions. It follows a fallback pattern where built-in
functions are checked first, then site-specific functions are retrieved from the
site-specific evaluator.
"""

import logging
from typing import Callable, Dict

from .models import AllocationResource
from pydantic import BaseModel, Field

from . import playwright_utils

from .types import (
    EvalFunc,
    StatusType,
    WebArenaSite,
    WebArenaTask,
    WebarenaTaskEvalResult,
    WebArenaTaskResponse,
)
from .datatype_evaluators import verify_retrieved_value_normalized
from .site_specific_evaluators import SiteSpecificEvaluator

logger = logging.getLogger(__name__)


class VerifyUiValueParams(BaseModel):
    locator: str = Field(..., description="The locator of the UI selector")
    url: str = Field(..., description="The URL of the page")


class VerifyUiSelectorValueParams(BaseModel):
    locator: str = Field(..., description="The locator of the UI selector")


class VerifyRetrievedValueNormalizedParams(BaseModel):
    """Pydantic model for eval_params structure of verify_retrieved_value_normalized function."""

    type: str = Field(
        ..., description="The type of validation: 'text', 'numeric', 'object', etc."
    )
    ordered: bool = Field(
        default=False, description="Whether order matters in validation"
    )


class EvaluatorFunctionRegistry:
    """
    Registry for evaluation functions that handles both built-in and site-specific functions.

    Built-in functions are registered by name and can be used across all sites.
    Site-specific functions are retrieved dynamically from the site-specific evaluator.
    """

    def __init__(self):
        """Initialize the registry with built-in functions."""
        self._builtin_functions: Dict[str, Callable] = {}
        self._site_specific_evaluator = SiteSpecificEvaluator()
        self._register_builtin_functions()

    def _register_builtin_functions(self):
        """Register all built-in evaluation functions."""
        # Common retrieve value functions (used across all sites)
        self._builtin_functions["verify_retrieved_value_normalized"] = (
            self._verify_retrieved_value_normalized
        )
        self._builtin_functions["validate_status"] = self._validate_status

        # Generic UI validation functions (used across sites)
        self._builtin_functions["verify_ui_selector_value"] = (
            self._verify_ui_selector_value
        )
        self._builtin_functions["verify_ui_contains_value"] = (
            self._verify_ui_contains_value
        )
        self._builtin_functions["verify_ui_form_value"] = self._verify_ui_form_value
        self._builtin_functions["verify_parent_url_path"] = self._verify_parent_url_path

    async def execute_function(
        self,
        eval_func: EvalFunc,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Execute an evaluation function by name.

        First checks for built-in functions, then falls back to site-specific functions.

        Args:
            eval_func: The evaluation function specification
            task: The WebArena task being evaluated
            task_result: The result from the test execution
            resource: The allocated resource used for the task
            **kwargs: Additional arguments

        Returns:
            WebarenaTaskEvalResult: The evaluation result
        """
        function_name = eval_func.name

        try:
            # First, check for built-in functions
            if function_name in self._builtin_functions:
                logger.info(f"Using built-in function: {function_name}")
                builtin_func = self._builtin_functions[function_name]

                # Prepare arguments for built-in function
                func_args = {
                    "eval_func": eval_func,
                    "task": task,
                    "task_result": task_result,
                    "resource": resource,
                    **kwargs,
                }

                # Execute built-in function
                result = await builtin_func(**func_args)
                return result

            # Fallback to site-specific functions via site-specific evaluator
            logger.info(f"Attempting site-specific function: {function_name}")
            result = await self._execute_site_specific_function(
                eval_func, task, task_result, resource, **kwargs
            )
            return result

        except Exception as e:
            logger.error(f"Error executing evaluation function '{function_name}': {e}")
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[f"Error executing evaluation function: {str(e)}"],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="function_execution_error",
            )

    async def _execute_site_specific_function(
        self,
        eval_func: EvalFunc,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Execute a site-specific function using the site-specific evaluator.

        Args:
            eval_func: The evaluation function specification
            task: The WebArena task being evaluated
            task_result: The result from the test execution
            resource: The allocated resource
            **kwargs: Additional arguments

        Returns:
            WebarenaTaskEvalResult: The evaluation result
        """
        # Determine the site from the resource
        site = WebArenaSite(resource.website_type)

        # Get the appropriate validator for this site
        validator = await self._site_specific_evaluator.get_validator_for_site(
            site, resource
        )
        if not validator:
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[f"No validator available for site: {site}"],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="site_specific_evaluation",
            )

        # Call the evaluation function directly on the validator
        result = await self._site_specific_evaluator.call_evaluation_function(
            validator=validator,
            function_name=eval_func.name,
            task=task,
            task_result=task_result,
            resource=resource,
            eval_func=eval_func,
            **kwargs,
        )

        return result

    # Built-in evaluation functions

    async def _verify_retrieved_value_normalized(
        self,
        eval_func: EvalFunc,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Verify the retrieved value is normalized according to the specified type.

        This function delegates to the helper function in evaluators.py to handle
        different types of value validation (text, number, etc.) and ordering.

        Args:
            eval_func: The evaluation function specification containing eval_params and expected_data
            task: The WebArena task being evaluated
            task_result: The result from the test execution
            resource: The allocated resource
            **kwargs: Additional arguments

        Returns:
            WebarenaTaskEvalResult: The evaluation result
        """

        # Check if eval_params exists
        if not hasattr(eval_func, "eval_params") or eval_func.eval_params is None:
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[
                    "Configuration error: eval_func.eval_params is missing for verify_retrieved_value_normalized function"
                ],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="value_validation",
            )

        # Check if expected_data exists
        if not hasattr(eval_func, "expected_data") or eval_func.expected_data is None:
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[
                    "Configuration error: eval_func.expected_data is missing for verify_retrieved_value_normalized function"
                ],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="value_validation",
            )

        try:
            # Validate eval_params using Pydantic model
            eval_params = VerifyRetrievedValueNormalizedParams.model_validate(
                eval_func.eval_params
            )
        except Exception as e:
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[
                    f"Configuration error: eval_params validation failed for verify_retrieved_value_normalized function - {str(e)}"
                ],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="value_validation",
            )

        # Delegate to helper function in evaluators.py
        return await verify_retrieved_value_normalized(
            task=task,
            task_result=task_result,
            resource=resource,
            eval_type=eval_params.type,
            ordered=eval_params.ordered,
            eval_func=eval_func,
            **kwargs,
        )

    async def _validate_status(
        self,
        eval_func: EvalFunc,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate the status of the task against expected status.

        Checks if the actual status from WebArenaTaskResponse matches
        the expected status specified in eval_func.expected_data.expected_status.

        Args:
            eval_func: The evaluation function specification containing expected_data
            task: The WebArena task being evaluated
            task_result: The result from the test execution
            resource: The allocated resource
            **kwargs: Additional arguments

        Returns:
            WebarenaTaskEvalResult: The evaluation result
        """
        # Check if expected_data exists
        if not hasattr(eval_func, "expected_data") or eval_func.expected_data is None:
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[
                    "Configuration error: eval_func.expected_data is missing for validate_status function"
                ],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="status_validation",
            )

        # Check if expected_status exists in expected_data
        if "expected_status" not in eval_func.expected_data:
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[
                    "Configuration error: expected_data.expected_status is missing for validate_status function"
                ],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="status_validation",
            )

        expected_status_str = eval_func.expected_data["expected_status"]
        if not task_result.response:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["Agent response not found in task result."],
                site=resource.website_type,
                task_id=task.task_id,
            )

        actual_status = task_result.response.status

        # Validate that expected_status is a valid StatusType value
        try:
            # Try to create StatusType from the expected string
            expected_status = StatusType(expected_status_str)
        except ValueError:
            # Get all valid StatusType values for error message
            valid_statuses = [status.value for status in StatusType]
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[
                    f"Configuration error: expected_status '{expected_status_str}' is not a valid StatusType. "
                    f"Valid values are: {', '.join(valid_statuses)}"
                ],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="status_validation",
            )

        # Compare the expected status with actual status
        if expected_status == actual_status:
            return WebarenaTaskEvalResult(
                score=1.0,
                assertion_msgs=[
                    f"Status validation passed: expected '{expected_status.value}' and got '{actual_status.value}'"
                ],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="status_validation",
            )
        else:
            return WebarenaTaskEvalResult(
                score=0.0,
                assertion_msgs=[
                    f"Status validation failed: expected '{expected_status.value}' but got '{actual_status.value}'"
                ],
                task_id=task.task_id,
                task_description=task.intent,
                task_type="status_validation",
            )

    async def _verify_ui_selector_value(
        self,
        eval_func: EvalFunc,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """Verify the UI selector value."""

        if not eval_func or eval_func.eval_params is None:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["No eval_params found in eval_func"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        try:
            eval_params = VerifyUiSelectorValueParams.model_validate(
                eval_func.eval_params
            )
        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        element_value, element_html = await playwright_utils.get_element_value(
            resource.cdp_url, eval_params.locator
        )

        validation_data = {
            "locator": eval_params.locator,
            "value": element_value,
            "element": element_html,
        }
        if element_value != eval_func.expected_data.get("value"):
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[
                    f"UI selector value verification failed: expected '{eval_func.expected_data.get('value')}', but got '{element_value}'"
                ],
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )
        else:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=[
                    f"UI selector value verification passed: expected '{eval_func.expected_data.get('value')}', got '{element_value}'"
                ],
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )

    async def _verify_ui_contains_value(
        self,
        eval_func: EvalFunc,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """Verify the UI value exists."""
        if not eval_func or eval_func.eval_params is None:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["No eval_params found in eval_func"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        try:
            eval_params = VerifyUiValueParams.model_validate(eval_func.eval_params)
        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        values, elements = await playwright_utils.get_element_values(
            resource.cdp_url, eval_params.locator
        )

        assert_msgs = []
        validation_failed = False

        expected_values = eval_func.expected_data["values"]
        normalized_values = [value.lower() for value in values]
        for expected_value in expected_values:
            found = False
            normalized_expected_value = expected_value.lower()
            for value in normalized_values:
                if normalized_expected_value in value:
                    found = True
                    break
            if not found:
                assert_msgs.append(f"Value '{expected_value}' not found in values")
                validation_failed = True
            else:
                assert_msgs.append(f"Value '{expected_value}' found in values")

        validation_data = {
            "locator": eval_params.locator,
            "values": values,
            "elements": elements,
        }

        if validation_failed:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assert_msgs,
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )
        else:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assert_msgs,
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )

    async def _verify_parent_url_path(
        self,
        eval_func: EvalFunc,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """Verify the parent URL path."""

        if not eval_func or eval_func.eval_params is None:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["No eval_params found in eval_func"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        expected_url = eval_func.eval_params.get("url")

        if not expected_url:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["No 'url' parameter found in eval_params"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        # Substitute website type placeholder with actual base URL
        cleaned_expected_url = expected_url.replace(
            f"__{resource.website_type.upper()}__", resource.base_url
        )

        if not task_result.last_urls:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["No URLs found in task_result.last_urls"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        # Check if any URL in last_urls starts with the expected_url
        matching_urls = []
        for url in task_result.last_urls:
            if url.startswith(cleaned_expected_url):
                matching_urls.append(url)

        validation_data = {
            "expected_url": expected_url,
            "cleaned_expected_url": cleaned_expected_url,
            "last_urls": task_result.last_urls,
            "matching_urls": matching_urls,
        }

        if matching_urls:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=[
                    f"Parent URL path verification passed: found {len(matching_urls)} URL(s) starting with '{cleaned_expected_url}'. Matching URLs: {matching_urls}"
                ],
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[
                    f"Parent URL path verification failed: no URLs in last_urls start with '{cleaned_expected_url}'. Actual URLs: {task_result.last_urls}"
                ],
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )

    async def _verify_ui_form_value(
        self,
        eval_func: EvalFunc,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """Verify the UI value exists."""
        if not eval_func or eval_func.eval_params is None:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["No eval_params found in eval_func"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        try:
            eval_params = VerifyUiValueParams.model_validate(eval_func.eval_params)
        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        form_fields = await playwright_utils.get_form_fields(
            resource.cdp_url, eval_params.locator
        )

        assert_msgs = []
        validation_failed = False

        def _process_field(expected_value, form_value) -> bool:
            validation_failed = False
            if expected_value not in form_value:
                assert_msgs.append(
                    f"Field '{field_name}' value mismatch: expected '{expected_value}', got '{form_value}'"
                )
                validation_failed = True
            else:
                assert_msgs.append(
                    f"Field '{field_name}' value match: expected '{expected_value}', got '{form_value}'"
                )
            return validation_failed

        for field_name, expected_value in eval_func.expected_data.items():
            if field_name not in form_fields:
                assert_msgs.append(f"Field '{field_name}' not found in form fields")
                validation_failed = True
            else:
                form_value = form_fields[field_name]
                if isinstance(expected_value, list):
                    for expected_value in expected_value:
                        validation_failed |= _process_field(expected_value, form_value)
                else:
                    validation_failed |= _process_field(expected_value, form_value)

        validation_data = {
            "locator": eval_params.locator,
            "form_fields": form_fields,
        }

        if validation_failed:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assert_msgs,
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )
        else:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assert_msgs,
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )


# Global registry instance
evaluator_function_registry = EvaluatorFunctionRegistry()


async def execute_eval_function(
    eval_func: EvalFunc,
    task: WebArenaTask,
    task_result: WebArenaTaskResponse,
    resource: AllocationResource,
    **kwargs,
) -> WebarenaTaskEvalResult:
    """
    Convenience function to execute an evaluation function.

    Args:
        eval_func: The evaluation function specification
        task: The WebArena task being evaluated
        task_result: The result from the test execution
        resource: The allocated resource used for the task
        **kwargs: Additional arguments

    Returns:
        WebarenaTaskEvalResult: The evaluation result
    """
    return await evaluator_function_registry.execute_function(
        eval_func=eval_func,
        task=task,
        task_result=task_result,
        resource=resource,
        **kwargs,
    )
