"""GitLab API service layer for evaluation purposes."""

import subprocess
import uuid
from pathlib import Path
from typing import Any
from urllib.parse import quote

import requests
from pydantic import Field, ValidationError, computed_field
from pydantic_settings import BaseSettings, SettingsConfigDict

from .models import (
    GitlabCommit,
    GitlabProject,
    GitlabProjectMember,
    GitlabRepositoryFile,
    Group,
    GroupMember,
    Issue,
    MergeRequest,
    MergeRequestNote,
    Milestone,
    StarredGitlabProject,
    User,
    UserDisplayStatus,
    UserProfile,
)


class GitLabSettings(BaseSettings):
    """Configuration for the GitLab API."""

    gitlab_username: str = "byteblaze"
    gitlab_host: str = "localhost"
    gitlab_port: int = 8500
    private_token: str = Field(
        "my-super-secret-1234", description="The token for authorization."
    )
    gitlab_version: str = Field(
        "15.7.5", description="The version of the GitLab instance."
    )
    gitlab_api_version: str = Field(
        "v4", description="Api version that this instance is running"
    )
    task_id: int | None = None
    log_task_results: bool = True

    model_config = SettingsConfigDict(
        env_prefix="WEBARENA_VERIFIED_GITLAB_",
        arbitrary_types_allowed=True,
    )

    @computed_field
    @property
    def gitlab_url(self) -> str:
        return f"http://{self.gitlab_host}:{self.gitlab_port}"

    @computed_field
    @property
    def gitlab_api_url(self) -> str:
        return f"http://{self.gitlab_host}:{self.gitlab_port}/api/{self.gitlab_api_version}"


class GitLabAPI:
    """A wrapper for the GitLab API."""

    def __init__(self, settings: GitLabSettings):
        self.settings = settings

    def get_user_projects(self, visibility=None) -> list[GitlabProject] | None:
        """Get current user's projects as a list of GitlabProject models.

        Args:
            visibility: Limit by visibility "public", "internal", or "private".

        """
        visibility = f"?visibility={visibility}" if visibility else ""
        url = f"{self.settings.gitlab_api_url}/users/{self.settings.gitlab_username}/projects{visibility}"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [GitlabProject.model_validate(item) for item in data]
        except Exception as e:
            print(f"Failed to parse projects: {e}")
            return None

    def get_user_starred_projects(
        self, *, user_id: int | str
    ) -> list[StarredGitlabProject] | None:
        """Get a user's starred projects.

        Args:
            user_id: The ID or username of the user.

        Returns:
            A list of StarredGitlabProject objects or None if not found.
        """
        url = f"{self.settings.gitlab_api_url}/users/{user_id}/starred_projects"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [StarredGitlabProject.model_validate(item) for item in data]
        except Exception as e:
            print(f"Failed to parse starred projects: {e}")
            return None

    def get_project(self, *, group: str, project: str) -> GitlabProject | None:
        """Get a project by group and project name.

        Args:
            group: The group or username.
            project: The project name.

        Returns:
            A GitlabProject object or None if not found.
        """
        project_path = f"{group}/{project}"
        encoded_project_path = quote(project_path, safe="")
        url = f"{self.settings.gitlab_api_url}/projects/{encoded_project_path}"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return GitlabProject.model_validate(data)
        except Exception as e:
            print(f"Failed to parse project: {e}")
            return None

    def get_project_members(
        self, *, project_id: int
    ) -> list[GitlabProjectMember] | None:
        """Get all members of a group as a list of GitlabGroupMember models."""
        url = f"{self.settings.gitlab_api_url}/projects/{project_id}/members"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [GitlabProjectMember.model_validate(item) for item in data]
        except Exception as e:
            print(f"Failed to parse group members: {e}")
            return None

    def get_groups(self) -> list[Group] | None:
        """Get all groups."""
        url = f"{self.settings.gitlab_api_url}/groups"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [Group.model_validate(item) for item in data]
        except Exception as e:
            print(f"Failed to parse groups: {e}")
            return None

    def get_group_members(self, *, group_id: int | str) -> list[GroupMember] | None:
        """Get all members of a group as a list of GroupMember models."""
        url = f"{self.settings.gitlab_api_url}/groups/{group_id}/members"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [GroupMember.model_validate(item) for item in data]
        except Exception as e:
            print(f"Failed to parse group members: {e}")
            return None

    def get_repository_file(
        self,
        *,
        project_id: int,
        file_path: str,
        branch: str = "main",
    ) -> GitlabRepositoryFile | None:
        """Get a file from a repository.

        Args:
            project_id: The ID of the project.
            file_path: The path to the file in the repository.
            branch: The name of the branch.

        Returns:
            A GitlabRepositoryFile object or None if not found.
        """
        # GitLab API requires the file path to be URL-encoded
        encoded_file_path = quote(file_path, safe="")
        url = f"{self.settings.gitlab_api_url}/projects/{project_id}/repository/files/{encoded_file_path}?ref={branch}"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return GitlabRepositoryFile.model_validate(data)
        except Exception as e:
            print(f"Failed to parse repository file: {e}")
            return None

    def get_issue(
        self,
        *,
        issue_id: int | None = None,
        search: str | None = None,
        in_param: str | None = None,
    ) -> Issue | list[Issue] | None:
        """Get an issue by its ID or a list of issues by query parameters.

        Args:
            issue_id: The ID of the issue.
            search: Search issues against their title and description.
            in_param: Modify the scope of the search attribute. 'title', 'description', or 'title,description'.

        Returns:
            An Issue object if issue_id is provided, otherwise a list of Issue objects or None.
        """
        if issue_id:
            url = f"{self.settings.gitlab_api_url}/issues/{issue_id}"
            data = self._make_gitlab_request(url)
            if data is None:
                return None
            try:
                return Issue.model_validate(data)
            except Exception as e:
                print(f"Failed to parse issue: {e}")
                return None
        else:
            params = {}
            if search:
                params["search"] = search
            if in_param:
                params["in"] = in_param

            url = f"{self.settings.gitlab_api_url}/issues"
            data = self._make_gitlab_request(url, params=params)
            if data is None:
                return None
            try:
                return [Issue.model_validate(item) for item in data]
            except Exception as e:
                print(f"Failed to parse issues: {e}")
                return None

    def get_issue_from_project(
        self, *, group: str, project: str, issue_iid: int
    ) -> Issue | None:
        """Get a single issue by IID within a specific project.

        Args:
            group: The group or username.
            project: The project name.
            issue_iid: The issue IID within the project.

        Returns:
            An Issue object or None if not found.
        """
        project_path = f"{group}/{project}"
        encoded_project_path = quote(project_path, safe="")
        url = f"{self.settings.gitlab_api_url}/projects/{encoded_project_path}/issues/{issue_iid}"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return Issue.model_validate(data)
        except Exception as e:
            print(f"Failed to parse issue: {e}")
            return None

    def get_project_issues(
        self, *, project_id: int | str, author_username: str | None = None
    ) -> list[Issue] | None:
        """Get all issues for a project.

        Args:
            project_id: The ID or URL-encoded path of the project.
            author_username: The username of the author to filter by.

        Returns:
            A list of Issue objects or None if not found.
        """
        if isinstance(project_id, str):
            project_id = quote(project_id, safe="")

        url = f"{self.settings.gitlab_api_url}/projects/{project_id}/issues"
        if author_username:
            url += f"?author_username={author_username}"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [Issue.model_validate(item) for item in data]
        except ValidationError as e:
            print(f"Failed to parse issues: {e.json()}")
            return None
        except Exception as e:
            print(f"An unexpected error occurred while parsing issues: {e}")
            return None

    def get_project_milestones(
        self,
        *,
        group: str,
        project: str,
        iids: list[int] | None = None,
        state: str | None = None,
        title: str | None = None,
        search: str | None = None,
        include_parent_milestones: bool | None = None,
    ) -> list[Milestone] | None:
        """Get all milestones for a project.

        Args:
            group: The group or username.
            project: The project name.
            iids: Return only the milestones having the given iid.
            state: Return only active or closed milestones.
            title: Return only the milestones having the given title.
            search: Return only milestones with a title or description matching the provided string.
            include_parent_milestones: Include group milestones from parent group and its ancestors.

        Returns:
            A list of Milestone objects or None if not found.
        """
        project_path = f"{group}/{project}"
        encoded_project_path = quote(project_path, safe="")
        url = (
            f"{self.settings.gitlab_api_url}/projects/{encoded_project_path}/milestones"
        )

        params = {}
        if iids:
            params["iids"] = iids
        if state:
            params["state"] = state
        if title:
            params["title"] = title
        if search:
            params["search"] = search
        if include_parent_milestones is not None:
            params["include_parent_milestones"] = include_parent_milestones

        data = self._make_gitlab_request(url, params=params)
        if data is None:
            return None
        try:
            return [Milestone.model_validate(item) for item in data]
        except ValidationError as e:
            print(f"Failed to parse milestones: {e.json()}")
            return None
        except Exception as e:
            print(f"An unexpected error occurred while parsing milestones: {e}")
            return None

    def get_merge_requests(
        self,
        *,
        group: str,
        project: str,
        author_username: str | None = None,
        reviewer_username: str | None = None,
        reviewer_id: int | None = None,
        search: str | None = None,
    ) -> list[MergeRequest] | None:
        """Get all merge requests for a project.

        Args:
            group: The group or username.
            project: The project name.
            author_username: The username of the author to filter by.
            reviewer_username: The username of the reviewer to filter by.
            reviewer_id: The ID of the reviewer to filter by.
            search: Search merge requests against their title.

        Returns:
            A list of MergeRequest objects or None if not found.
        """
        project_details = self.get_project(group=group, project=project)
        if not project_details:
            return None

        url = f"{self.settings.gitlab_api_url}/projects/{project_details.id}/merge_requests"
        params = {}
        if author_username:
            params["author_username"] = author_username
        if reviewer_username:
            params["reviewer_username"] = reviewer_username
        if reviewer_id:
            params["reviewer_id"] = reviewer_id
        if search:
            params["search"] = search
            params["in"] = "title"

        data = self._make_gitlab_request(url, params=params)
        if data is None:
            return None
        try:
            return [MergeRequest.model_validate(item) for item in data]
        except ValidationError as e:
            print(f"Failed to parse merge requests: {e.json()}")
            return None
        except Exception as e:
            print(f"An unexpected error occurred while parsing merge requests: {e}")
            return None

    def get_merge_request_notes(
        self, *, group: str, project: str, merge_request_iid: int
    ) -> list[MergeRequestNote] | None:
        """Get all notes for a merge request.

        Args:
            group: The group or username.
            project: The project name.
            merge_request_iid: The IID of the merge request.

        Returns:
            A list of MergeRequestNote objects or None if not found.
        """
        project_path = f"{group}/{project}"
        encoded_project_path = quote(project_path, safe="")
        url = f"{self.settings.gitlab_api_url}/projects/{encoded_project_path}/merge_requests/{merge_request_iid}/notes"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [MergeRequestNote.model_validate(item) for item in data]
        except Exception as e:
            print(f"Failed to parse merge request notes: {e}")
            return None

    def get_project_commits(
        self, *, project_id: int | str
    ) -> list[GitlabCommit] | None:
        """Get all commits for a project.

        Args:
            project_id: The ID or URL-encoded path of the project.

        Returns:
            A list of GitlabCommit objects or None if not found.
        """
        if isinstance(project_id, str):
            project_id = quote(project_id, safe="")

        url = f"{self.settings.gitlab_api_url}/projects/{project_id}/repository/commits"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [GitlabCommit.model_validate(item) for item in data]
        except ValidationError as e:
            print(f"Failed to parse commits: {e.json()}")
            return None
        except Exception as e:
            print(f"An unexpected error occurred while parsing commits: {e}")
            return None

    def get_user_by_username(self, *, username: str) -> User | None:
        """Get a user by username.

        Args:
            username: The username to search for.

        Returns:
            A User object or None if not found.
        """
        url = f"{self.settings.gitlab_api_url}/users?username={username}"
        data = self._make_gitlab_request(url)
        if not data:
            return None
        try:
            # Assuming the first result is the correct one
            return User.model_validate(data[0])
        except Exception as e:
            print(f"Failed to parse user: {e}")
            return None

    def get_user_following(self, *, user_id: int) -> list[User] | None:
        """Get the users that a user is following.

        Args:
            user_id: The ID of the user.

        Returns:
            A list of User objects or None if not found.
        """
        url = f"{self.settings.gitlab_api_url}/users/{user_id}/following"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return [User.model_validate(item) for item in data]
        except Exception as e:
            print(f"Failed to parse users: {e}")
            return None

    def get_user_profile(self, *, user_id: int) -> UserProfile | None:
        """Get a user's profile by user ID.

        Args:
            user_id: The ID of the user.

        Returns:
            A UserProfile object or None if not found.
        """
        url = f"{self.settings.gitlab_api_url}/users/{user_id}"
        data = self._make_gitlab_request(url)
        if not data:
            return None
        try:
            return UserProfile.model_validate(data)
        except Exception as e:
            print(f"Failed to parse user profile: {e}")
            return None

    def get_user_status(self, *, user_id: int | str) -> UserDisplayStatus | None:
        """Get a user's display status.

        Args:
            user_id: The ID or username of the user.

        Returns:
            A UserDisplayStatus object or None if not found.
        """
        url = f"{self.settings.gitlab_api_url}/users/{user_id}/status"
        data = self._make_gitlab_request(url)
        if data is None:
            return None
        try:
            return UserDisplayStatus.model_validate(data)
        except ValidationError as e:
            print(f"Failed to parse user status: {e}")
            return None

    def _admin_create_personal_access_token(self) -> str:
        """Create and return a personal access token with highest permissions for testing web arena"""
        token_scopes = (
            "[:api, :read_api, :read_user, :read_repository, :write_repository]"
        )
        uid = str(uuid.uuid4())[:4]
        token_name = f"Automation Token {uid}"
        token_value = f"my-super-secret-{uid}"
        command = [
            "docker",
            "exec",
            "-it",
            "gitlab",
            "gitlab-rails",
            "runner",
            f"token = User.find_by_username('{self.settings.gitlab_username}').personal_access_tokens.create(scopes: {token_scopes}, name: '{token_name}'); token.set_token('{token_value}'); token.save!",
        ]
        print("Creating token. One moment...")
        result = subprocess.run(command, capture_output=True, text=True)
        print(result.stdout)
        print(result.stderr)
        return token_value

    def _make_gitlab_request(self, url: str, params: dict | None = None) -> Any | None:
        """Make a GitLab API request with error handling."""
        try:
            response = requests.get(
                url,
                headers={"PRIVATE-TOKEN": self.settings.private_token},
                params=params,
            )
            response.raise_for_status()  # Raises HTTPError for bad responses
            if response.status_code == 204:  # No content
                return None
            return response.json()
        except requests.exceptions.HTTPError as e:
            if e.response.status_code == 404:
                return None  # Resource not found
            raise e
        except requests.exceptions.RequestException as e:
            print(f"Request failed: {e}")
            return None
