from __future__ import annotations

from datetime import datetime
from enum import IntEnum, StrEnum
from typing import List, Optional

from pydantic import BaseModel, ConfigDict, Field


class GitLabVisibility(StrEnum):
    PUBLIC = "public"
    INTERNAL = "internal"
    PRIVATE = "private"


class GitLabAccessLevel(IntEnum):
    NO_ACCESS = 0
    MINIMAL_ACCESS = 5
    GUEST = 10
    PLANNER = 15
    REPORTER = 20
    DEVELOPER = 30
    MAINTAINER = 40
    OWNER = 50


class GitLabBaseModel(BaseModel):
    model_config = ConfigDict(json_encoders={datetime: lambda dt: dt.isoformat()})


class Namespace(GitLabBaseModel):
    id: int
    name: str
    path: str
    kind: str
    full_path: str
    parent_id: Optional[int] = None
    avatar_url: Optional[str] = None
    web_url: str


class GitlabProject(GitLabBaseModel):
    id: int
    description: Optional[str] = None
    name: str
    name_with_namespace: str
    path: str
    path_with_namespace: str
    created_at: datetime
    default_branch: str
    tag_list: List[str]
    topics: List[str]
    ssh_url_to_repo: str
    http_url_to_repo: str
    web_url: str
    readme_url: Optional[str] = None
    avatar_url: Optional[str] = None
    forks_count: int
    star_count: int
    last_activity_at: datetime
    namespace: Namespace
    visibility: Optional[GitLabVisibility] = None


class GitlabMemberCreatedBy(GitLabBaseModel):
    id: int
    username: str
    name: str
    state: str
    avatar_url: Optional[str] = None
    web_url: str


class GitlabProjectMember(GitLabBaseModel):
    id: int
    username: str
    name: str
    state: str
    avatar_url: Optional[str] = None
    web_url: str
    access_level: int
    created_at: datetime
    created_by: Optional[GitlabMemberCreatedBy] = None
    expires_at: Optional[datetime] = None


class GitlabRepositoryFile(GitLabBaseModel):
    file_name: str
    file_path: str
    size: int
    encoding: str
    content: str
    content_sha256: str
    ref: str
    blob_id: str
    commit_id: str
    last_commit_id: str


class Milestone(GitLabBaseModel):
    id: int
    iid: int
    project_id: int
    title: str
    description: str
    due_date: Optional[str] = None
    start_date: Optional[str] = None
    state: str
    updated_at: datetime
    created_at: datetime
    closed_at: Optional[datetime] = None
    expired: Optional[bool] = None
    web_url: Optional[str] = None


class User(GitLabBaseModel):
    id: int
    username: str
    name: str
    state: str
    avatar_url: str | None = None
    web_url: str | None = None


class UserProfile(User):
    created_at: datetime
    bio: str | None = None
    location: str | None = None
    public_email: str | None = None
    skype: str | None = None
    linkedin: str | None = None
    twitter: str | None = None
    website_url: str | None = None
    organization: str | None = None
    job_title: str | None = None
    pronouns: str | None = None
    bot: bool | None = None
    work_information: str | None = None
    followers: int | None = None
    following: int | None = None
    is_followed: bool | None = None
    local_time: str | None = None


class UserDisplayStatus(GitLabBaseModel):
    emoji: Optional[str] = None
    message: Optional[str] = ""
    availability: Optional[str] = None
    message_html: str
    clear_status_at: Optional[datetime] = None


class References(GitLabBaseModel):
    short: str
    relative: str
    full: str


class TimeStats(GitLabBaseModel):
    time_estimate: int
    total_time_spent: int
    human_time_estimate: Optional[str] = None
    human_total_time_spent: Optional[str] = None


class TaskCompletionStatus(GitLabBaseModel):
    count: int
    completed_count: int


class Links(GitLabBaseModel):
    self: str
    notes: str
    award_emoji: str
    project: str
    closed_as_duplicate_of: Optional[str] = None


class GitlabCommit(GitLabBaseModel):
    id: str
    short_id: str
    created_at: datetime
    parent_ids: List[str]
    title: str
    message: str
    author_name: str
    author_email: str
    authored_date: datetime
    committer_name: str
    committer_email: str
    committed_date: datetime
    trailers: Optional[dict] = None
    web_url: str


class MergeRequestNoteAuthor(GitLabBaseModel):
    id: int
    username: str
    email: Optional[str] = None
    name: str
    state: str
    created_at: Optional[datetime] = None


class MergeRequestNote(GitLabBaseModel):
    id: int
    body: str
    attachment: Optional[str] = None
    author: MergeRequestNoteAuthor
    created_at: datetime
    updated_at: datetime
    system: bool
    noteable_id: int
    noteable_type: str
    noteable_iid: int
    resolvable: bool
    confidential: bool
    internal: bool


class MergeRequest(GitLabBaseModel):
    id: int
    iid: int
    project_id: int
    title: str
    description: Optional[str] = None
    state: str
    merged_by: Optional[User] = None
    merge_user: Optional[User] = None
    merged_at: Optional[datetime] = None
    closed_by: Optional[User] = None
    closed_at: Optional[datetime] = None
    created_at: datetime
    updated_at: datetime
    target_branch: str
    source_branch: str
    upvotes: int
    downvotes: int
    author: User
    assignee: Optional[User] = None
    assignees: List[User]
    reviewers: List[User]
    source_project_id: int
    target_project_id: int
    labels: List[str]
    draft: bool
    work_in_progress: bool
    milestone: Optional[Milestone] = None
    merge_when_pipeline_succeeds: bool
    merge_status: str
    detailed_merge_status: str
    sha: str
    merge_commit_sha: Optional[str] = None
    squash_commit_sha: Optional[str] = None
    user_notes_count: int
    discussion_locked: Optional[bool] = None
    should_remove_source_branch: Optional[bool] = None
    force_remove_source_branch: Optional[bool] = None
    allow_collaboration: Optional[bool] = None
    allow_maintainer_to_push: Optional[bool] = None
    web_url: str
    references: References
    time_stats: TimeStats
    squash: bool
    task_completion_status: TaskCompletionStatus


class Issue(GitLabBaseModel):
    id: int
    milestone: Optional[Milestone] = None
    author: User
    description: Optional[str] = None
    state: str
    iid: int
    assignees: List[User]
    assignee: Optional[User] = None
    type: str
    labels: List[str]
    upvotes: int
    downvotes: int
    merge_requests_count: int
    title: str
    updated_at: datetime
    created_at: datetime
    closed_at: Optional[datetime] = None
    closed_by: Optional[User] = None
    subscribed: Optional[bool] = None
    user_notes_count: int
    due_date: Optional[str] = None
    web_url: str
    references: References
    time_stats: TimeStats
    confidential: bool
    discussion_locked: Optional[bool] = None
    issue_type: str
    severity: str
    task_completion_status: TaskCompletionStatus
    weight: Optional[int] = None
    has_tasks: bool
    links: Links = Field(..., alias="_links")
    moved_to_id: Optional[int] = None
    service_desk_reply_to: Optional[str] = None


class ProjectAccess(GitLabBaseModel):
    access_level: int
    notification_level: int


class Permissions(GitLabBaseModel):
    project_access: Optional[ProjectAccess] = None
    group_access: Optional[dict] = None


class ContainerExpirationPolicy(GitLabBaseModel):
    cadence: str
    enabled: bool
    keep_n: int
    older_than: str
    name_regex: Optional[str] = None
    name_regex_keep: Optional[str] = None
    next_run_at: datetime


class ProjectLinks(GitLabBaseModel):
    self: str
    issues: str
    merge_requests: str
    repo_branches: str
    labels: str
    events: str
    members: str
    cluster_agents: str


class GroupMember(GitLabBaseModel):
    id: int
    username: str
    name: str
    state: str
    avatar_url: Optional[str] = None
    web_url: str
    access_level: int
    created_at: datetime
    created_by: Optional[GitlabMemberCreatedBy] = None
    expires_at: Optional[datetime] = None


class Group(GitLabBaseModel):
    id: int
    web_url: str
    name: str
    path: str
    description: str
    visibility: str
    share_with_group_lock: bool
    require_two_factor_authentication: bool
    two_factor_grace_period: int
    project_creation_level: str
    auto_devops_enabled: Optional[bool] = None
    subgroup_creation_level: str
    emails_disabled: Optional[bool] = None
    mentions_disabled: Optional[bool] = None
    lfs_enabled: bool
    default_branch_protection: int
    avatar_url: Optional[str] = None
    request_access_enabled: bool
    full_name: str
    full_path: str
    created_at: datetime
    parent_id: Optional[int] = None


class StarredGitlabProject(GitLabBaseModel):
    id: int
    description: Optional[str] = None
    name: str
    name_with_namespace: str
    path: str
    path_with_namespace: str
    created_at: datetime
    default_branch: str
    tag_list: List[str]
    topics: List[str]
    ssh_url_to_repo: str
    http_url_to_repo: str
    web_url: str
    readme_url: Optional[str] = None
    avatar_url: Optional[str] = None
    forks_count: int
    star_count: int
    last_activity_at: datetime
    namespace: Namespace
    links: ProjectLinks = Field(..., alias="_links")
    packages_enabled: Optional[bool] = None
    empty_repo: bool
    archived: bool
    visibility: str
    owner: Optional[User] = None
    resolve_outdated_diff_discussions: Optional[bool] = None
    container_expiration_policy: Optional[ContainerExpirationPolicy] = None
    issues_enabled: bool
    merge_requests_enabled: bool
    wiki_enabled: bool
    jobs_enabled: bool
    snippets_enabled: bool
    container_registry_enabled: bool
    service_desk_enabled: bool
    service_desk_address: Optional[str] = None
    can_create_merge_request_in: Optional[bool] = None
    issues_access_level: str
    repository_access_level: str
    merge_requests_access_level: str
    forking_access_level: str
    wiki_access_level: str
    builds_access_level: str
    snippets_access_level: str
    pages_access_level: str
    operations_access_level: str
    analytics_access_level: str
    container_registry_access_level: str
    security_and_compliance_access_level: str
    releases_access_level: str
    environments_access_level: str
    feature_flags_access_level: str
    infrastructure_access_level: str
    monitor_access_level: str
    emails_disabled: Optional[bool] = None
    shared_runners_enabled: bool
    lfs_enabled: bool
    creator_id: int
    import_url: Optional[str] = None
    import_type: Optional[str] = None
    import_status: Optional[str] = None
    open_issues_count: int
    ci_default_git_depth: Optional[int] = None
    ci_forward_deployment_enabled: Optional[bool] = None
    ci_job_token_scope_enabled: bool
    ci_separated_caches: bool
    ci_opt_in_jwt: bool
    ci_allow_fork_pipelines_to_run_in_parent_project: bool
    public_jobs: bool
    build_timeout: int
    auto_cancel_pending_pipelines: str
    ci_config_path: Optional[str] = None
    shared_with_groups: List
    only_allow_merge_if_pipeline_succeeds: bool
    allow_merge_on_skipped_pipeline: Optional[bool] = None
    restrict_user_defined_variables: bool
    request_access_enabled: bool
    only_allow_merge_if_all_discussions_are_resolved: bool
    remove_source_branch_after_merge: bool
    printing_merge_request_link_enabled: bool
    merge_method: str
    squash_option: str
    enforce_auth_checks_on_uploads: bool
    suggestion_commit_message: Optional[str] = None
    merge_commit_template: Optional[str] = None
    squash_commit_template: Optional[str] = None
    issue_branch_template: Optional[str] = None
    auto_devops_enabled: bool
    auto_devops_deploy_strategy: str
    autoclose_referenced_issues: bool
    keep_latest_artifact: bool
    runner_token_expiration_interval: Optional[int] = None
    permissions: Optional[Permissions] = None
